/* global browser, chrome */
// Browser compatibility layer
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

// Auth helper - Get headers with API key or userId as fallback
async function getAuthHeaders() {
  const { apiKey, userId } = await browserAPI.storage.local.get(['apiKey', 'userId']);
  const headers = {
    'Content-Type': 'application/json'
  };
  // Use API key if available, otherwise use userId as Bearer token
  if (apiKey) {
    headers['Authorization'] = `Bearer ${apiKey}`;
  } else if (userId) {
    headers['Authorization'] = `Bearer ${userId}`;
  }
  return headers;
}

// Threads UI JavaScript
// Fetches and displays conversation threads from the API

const API_URL = 'https://api.safehaiven.com/api';

// Platform colors and icons
const PLATFORM_CONFIG = {
  ChatGPT: { icon: '🟢', class: 'chatgpt' },
  Claude: { icon: '🟠', class: 'claude' },
  Grok: { icon: '⚪', class: 'grok' },
  Perplexity: { icon: '🔵', class: 'perplexity' },
  Gemini: { icon: '🔷', class: 'gemini' }
};

document.addEventListener('DOMContentLoaded', async () => {
  await loadThreads();

  // Refresh button
  document.getElementById('refreshThreads').addEventListener('click', async () => {
    await loadThreads();
  });
});

async function loadThreads() {
  const threadsList = document.getElementById('threadsList');
  threadsList.textContent = 'Loading conversation threads...'; // SECURITY: textContent
  threadsList.className = 'loading';

  try {
    // Get userId from storage
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    if (!userId) {
      // SECURITY: Build empty state with safe DOM
      const emptyState = document.createElement('div');
      emptyState.className = 'empty-state';

      const icon = document.createElement('div');
      icon.className = 'empty-icon';
      icon.textContent = '🔒';

      const title = document.createElement('div');
      title.className = 'empty-title';
      title.textContent = 'User ID Not Set';

      const text = document.createElement('div');
      text.className = 'empty-text';
      text.textContent = 'Please set your User ID in settings to view conversation threads.';

      emptyState.appendChild(icon);
      emptyState.appendChild(title);
      emptyState.appendChild(text);

      threadsList.textContent = ''; // Clear first
      threadsList.appendChild(emptyState);
      return;
    }

    // Fetch threads from API
    const response = await fetch(`${API_URL}/threads/user/${userId}?limit=10`, {
      headers: await getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error(`API returned ${response.status}`);
    }

    const data = await response.json();
    console.log('Threads data:', data);

    if (!data.success || !data.threads || data.threads.length === 0) {
      // SECURITY: Build empty state with safe DOM
      const emptyState = document.createElement('div');
      emptyState.className = 'empty-state';

      const icon = document.createElement('div');
      icon.className = 'empty-icon';
      icon.textContent = '💬';

      const title = document.createElement('div');
      title.className = 'empty-title';
      title.textContent = 'No Conversations Yet';

      const text = document.createElement('div');
      text.className = 'empty-text';
      text.textContent =
        'Start using AI assistants across platforms to create conversation threads.';

      emptyState.appendChild(icon);
      emptyState.appendChild(title);
      emptyState.appendChild(text);

      threadsList.textContent = ''; // Clear first
      threadsList.appendChild(emptyState);
      return;
    }

    // Display threads
    threadsList.textContent = ''; // SECURITY: clear with textContent
    data.threads.forEach((thread) => {
      const threadCard = createThreadCard(thread);
      threadsList.appendChild(threadCard);
    });
  } catch (error) {
    console.error('Error loading threads:', error);
    // SECURITY: Build error state with safe DOM
    const emptyState = document.createElement('div');
    emptyState.className = 'empty-state';

    const icon = document.createElement('div');
    icon.className = 'empty-icon';
    icon.textContent = '⚠️';

    const title = document.createElement('div');
    title.className = 'empty-title';
    title.textContent = 'Failed to Load Threads';

    const text = document.createElement('div');
    text.className = 'empty-text';
    text.textContent = error.message; // SAFE: error message via textContent

    emptyState.appendChild(icon);
    emptyState.appendChild(title);
    emptyState.appendChild(text);

    threadsList.textContent = ''; // Clear first
    threadsList.appendChild(emptyState);
  }
}

function createThreadCard(thread) {
  const card = document.createElement('div');
  card.className = 'thread-card';

  // Thread header
  const header = document.createElement('div');
  header.className = 'thread-header';

  const title = document.createElement('div');
  title.className = 'thread-title';
  title.textContent = thread.title || 'Untitled Conversation';

  const count = document.createElement('div');
  count.className = 'thread-count';
  count.textContent = `${thread.memory_count} ${thread.memory_count === 1 ? 'message' : 'messages'}`;

  header.appendChild(title);
  header.appendChild(count);

  // Platform flow
  if (thread.platforms && thread.platforms.length > 0) {
    const flow = document.createElement('div');
    flow.className = 'platform-flow';

    thread.platforms.forEach((platform, index) => {
      const badge = createPlatformBadge(platform);
      flow.appendChild(badge);

      // Add arrow between platforms
      if (index < thread.platforms.length - 1) {
        const arrow = document.createElement('span');
        arrow.className = 'flow-arrow';
        arrow.textContent = '→';
        flow.appendChild(arrow);
      }
    });

    card.appendChild(header);
    card.appendChild(flow);
  } else {
    card.appendChild(header);
  }

  // Preview (first memory)
  if (thread.memories && thread.memories.length > 0) {
    const preview = document.createElement('div');
    preview.className = 'thread-preview';
    preview.textContent = thread.memories[0].content;
    card.appendChild(preview);
  }

  // Meta info
  const meta = document.createElement('div');
  meta.className = 'thread-meta';

  const timeSpan = document.createElement('div');
  timeSpan.className = 'thread-time';
  timeSpan.textContent = `⏱️ ${formatTime(thread.updated_at)}`; // SECURITY: textContent

  meta.appendChild(timeSpan);

  if (thread.primary_world) {
    const worldBadge = document.createElement('span');
    worldBadge.textContent = thread.primary_world;
    worldBadge.style.fontSize = '11px';
    worldBadge.style.color = 'var(--text-tertiary)';
    meta.appendChild(worldBadge);
  }

  card.appendChild(meta);

  // Click to expand thread details
  card.addEventListener('click', async () => {
    await showThreadDetails(thread.id);
  });

  return card;
}

function createPlatformBadge(platform) {
  const badge = document.createElement('div');
  badge.className = 'platform-badge';

  const config = PLATFORM_CONFIG[platform] || { icon: '⚪', class: 'default' };
  badge.classList.add(config.class);

  // SECURITY: Build badge safely
  const iconSpan = document.createElement('span');
  iconSpan.textContent = config.icon;
  const textSpan = document.createElement('span');
  textSpan.textContent = platform; // SAFE: platform name from API
  badge.appendChild(iconSpan);
  badge.appendChild(textSpan);

  return badge;
}

function formatTime(timestamp) {
  const date = new Date(timestamp);
  const now = new Date();
  const diffMs = now - date;
  const diffMins = Math.floor(diffMs / 60000);
  const diffHours = Math.floor(diffMs / 3600000);
  const diffDays = Math.floor(diffMs / 86400000);

  if (diffMins < 1) return 'just now';
  if (diffMins < 60) return `${diffMins}m ago`;
  if (diffHours < 24) return `${diffHours}h ago`;
  if (diffDays < 7) return `${diffDays}d ago`;

  return date.toLocaleDateString();
}

async function showThreadDetails(threadId) {
  const threadsList = document.getElementById('threadsList');
  threadsList.textContent = 'Loading thread details...'; // SECURITY: textContent
  threadsList.className = 'loading';

  try {
    const response = await fetch(`${API_URL}/threads/${threadId}`, {
      headers: await getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error(`API returned ${response.status}`);
    }

    const data = await response.json();

    if (!data.success || !data.thread) {
      throw new Error('Thread not found');
    }

    // Display full thread
    const container = document.createElement('div');

    // Back button
    const backBtn = document.createElement('button');
    backBtn.className = 'back-btn';
    backBtn.textContent = '← Back to Threads';
    backBtn.addEventListener('click', () => loadThreads());
    container.appendChild(backBtn);

    // Thread header
    const header = document.createElement('div');
    header.style.marginBottom = 'var(--space-lg)';

    const title = document.createElement('h2');
    title.style.fontSize = '18px';
    title.style.fontWeight = '600';
    title.style.marginBottom = '8px';
    title.textContent = data.thread.title || 'Untitled Conversation';

    const subtitle = document.createElement('div');
    subtitle.style.fontSize = '13px';
    subtitle.style.color = 'var(--text-secondary)';
    subtitle.textContent = `${data.thread.memory_count} messages across ${data.thread.platforms.length} platforms`;

    header.appendChild(title);
    header.appendChild(subtitle);
    container.appendChild(header);

    // Memories in thread
    const expanded = document.createElement('div');
    expanded.className = 'thread-expanded';

    data.thread.memories.forEach((memory, index) => {
      const memCard = document.createElement('div');
      memCard.className = 'memory-in-thread';

      const platform = document.createElement('div');
      platform.className = 'memory-platform';
      const platformName = memory.metadata?.platform || 'Unknown';
      const config = PLATFORM_CONFIG[platformName] || { icon: '⚪' };
      platform.textContent = `${config.icon} ${platformName}`; // SECURITY: textContent
      platform.style.color = getPlatformColor(platformName);

      const text = document.createElement('div');
      text.className = 'memory-text';
      text.textContent = memory.content;

      memCard.appendChild(platform);
      memCard.appendChild(text);
      expanded.appendChild(memCard);

      // Add visual connector (except for last item)
      if (index < data.thread.memories.length - 1) {
        const connector = document.createElement('div');
        connector.style.width = '2px';
        connector.style.height = '12px';
        connector.style.background = 'var(--border-subtle)';
        connector.style.marginLeft = '10px';
        connector.style.marginBottom = '4px';
        expanded.appendChild(connector);
      }
    });

    container.appendChild(expanded);

    threadsList.textContent = ''; // SECURITY: clear with textContent
    threadsList.appendChild(container);
  } catch (error) {
    console.error('Error loading thread details:', error);
    // SECURITY: Build error state with safe DOM
    const emptyState = document.createElement('div');
    emptyState.className = 'empty-state';

    const icon = document.createElement('div');
    icon.className = 'empty-icon';
    icon.textContent = '⚠️';

    const errorTitle = document.createElement('div'); // Renamed to avoid conflict with 'title' in try block
    errorTitle.className = 'empty-title';
    errorTitle.textContent = 'Failed to Load Thread';

    const text = document.createElement('div');
    text.className = 'empty-text';
    text.textContent = error.message; // SAFE: error message via textContent

    emptyState.appendChild(icon);
    emptyState.appendChild(errorTitle);
    emptyState.appendChild(text);

    threadsList.textContent = ''; // Clear first
    threadsList.appendChild(emptyState);

    // Add back button on error
    setTimeout(() => {
      const backBtn = document.createElement('button');
      backBtn.className = 'back-btn';
      backBtn.textContent = '← Back to Threads';
      backBtn.addEventListener('click', () => loadThreads());
      threadsList.insertBefore(backBtn, threadsList.firstChild);
    }, 100);
  }
}

function getPlatformColor(platform) {
  const colors = {
    ChatGPT: '#10a37f',
    Claude: '#cc8f66',
    Grok: '#fff',
    Perplexity: '#20bbff',
    Gemini: '#4285f4'
  };
  return colors[platform] || 'var(--text-primary)';
}
