#!/usr/bin/env node
/**
 * Haiven Dashboard - Extensive UX Test Suite
 *
 * Tests all user flows from an average user's perspective.
 *
 * Usage:
 *   node test-ux-extensive.js --headed     # Watch tests run
 *   node test-ux-extensive.js --local      # Test localhost
 *   node test-ux-extensive.js --slow       # Slow mode for debugging
 */

import puppeteer from 'puppeteer';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const args = process.argv.slice(2);
const HEADED = args.includes('--headed');
const LOCAL = args.includes('--local');
const SLOW = args.includes('--slow');

const BASE_URL = LOCAL ? 'http://localhost:3001' : 'https://www.safehaiven.com';
const DASHBOARD_URL = `${BASE_URL}/dashboard.html`;
const DELAY = SLOW ? 2000 : 300;
const EXTENSION_PATH = path.resolve(__dirname);

// Test results tracking
const results = { passed: [], failed: [], skipped: [] };
let screenshotCount = 0;

// Sample conversation for paste testing
const SAMPLE_CONVERSATION = [
  'User: I want to start a podcast about productivity. Where should I host it?',
  '',
  'Assistant: Great idea! Here are my top recommendations:',
  '',
  '1. **Anchor (Spotify)** - Free, easy to use, automatic distribution',
  '2. **Buzzsprout** - Best analytics, $12/month',
  '3. **Transistor** - Great for multiple shows, $19/month',
  '',
  'For a productivity podcast, I recommend starting with Anchor since it is free and handles distribution automatically.',
  '',
  'User: I like Anchor. What equipment do I need?',
  '',
  'Assistant: For a beginner setup:',
  '',
  '- **Microphone**: Audio-Technica ATR2100x ($99) or Blue Yeti ($130)',
  '- **Headphones**: Any closed-back headphones ($30-50)',
  '- **Pop filter**: Basic one for $10-15',
  '',
  'Decision: Start with Anchor for hosting.',
  'To-do: Order the Audio-Technica microphone this week.'
].join('\n');

// Logging utilities
function log(msg, type = 'info') {
  const colors = {
    info: '\x1b[36m',
    pass: '\x1b[32m',
    fail: '\x1b[31m',
    warn: '\x1b[33m',
    section: '\x1b[35m',
    reset: '\x1b[0m'
  };
  const prefix = {
    info: '[INFO]',
    pass: '[PASS]',
    fail: '[FAIL]',
    warn: '[WARN]',
    section: '\n[====]'
  };
  console.log(`${colors[type]}${prefix[type]}${colors.reset} ${msg}`);
}

// Test wrapper
async function test(name, fn, page) {
  try {
    await fn();
    results.passed.push(name);
    log(name, 'pass');
  } catch (err) {
    results.failed.push({ name, error: err.message });
    log(`${name}: ${err.message}`, 'fail');
    // Take screenshot on failure
    if (page) {
      screenshotCount++;
      const screenshotPath = `/tmp/haiven-test-fail-${screenshotCount}.png`;
      await page.screenshot({ path: screenshotPath });
      log(`Screenshot saved: ${screenshotPath}`, 'warn');
    }
  }
}

// Wait helper
const wait = (ms) => new Promise((r) => setTimeout(r, ms));

// Main test suite
async function runTests() {
  console.log('\n' + '='.repeat(60));
  log('HAIVEN DASHBOARD - EXTENSIVE UX TEST SUITE', 'section');
  console.log('='.repeat(60));
  log(`URL: ${DASHBOARD_URL}`);
  log(`Mode: ${HEADED ? 'Headed (visible)' : 'Headless'}`);
  log(`Speed: ${SLOW ? 'Slow (debugging)' : 'Normal'}`);
  console.log('');

  const browser = await puppeteer.launch({
    headless: !HEADED,
    args: [
      `--disable-extensions-except=${EXTENSION_PATH}`,
      `--load-extension=${EXTENSION_PATH}`,
      '--no-sandbox',
      '--disable-setuid-sandbox'
    ],
    defaultViewport: { width: 1400, height: 900 }
  });

  const page = await browser.newPage();

  // Track console errors
  const consoleErrors = [];
  page.on('console', (msg) => {
    if (msg.type() === 'error') {
      consoleErrors.push(msg.text());
    }
  });

  try {
    // ========================================
    // SECTION 1: PAGE LOAD & FIRST IMPRESSION
    // ========================================
    log('SECTION 1: Page Load & First Impression', 'section');

    await test('1.1 Dashboard loads without error', async () => {
      const response = await page.goto(DASHBOARD_URL, {
        waitUntil: 'networkidle2',
        timeout: 30000
      });
      if (!response.ok()) {
        throw new Error(`HTTP ${response.status()}`);
      }
    }, page);

    await wait(DELAY);

    await test('1.2 Page title is correct', async () => {
      const title = await page.title();
      if (!title.toLowerCase().includes('haiven')) {
        throw new Error(`Unexpected title: "${title}"`);
      }
    }, page);

    await test('1.3 No console errors on load', async () => {
      const criticalErrors = consoleErrors.filter(
        (e) => !e.includes('favicon') && !e.includes('404')
      );
      if (criticalErrors.length > 0) {
        throw new Error(`${criticalErrors.length} console errors: ${criticalErrors[0]}`);
      }
    }, page);

    await test('1.4 Main content area is visible', async () => {
      const content = await page.$('.threads-dashboard, .dashboard, main, [role="main"]');
      if (!content) {
        throw new Error('No main content area found');
      }
    }, page);

    // ========================================
    // SECTION 2: NAVIGATION
    // ========================================
    log('SECTION 2: Navigation', 'section');

    await test('2.1 Top navigation bar exists', async () => {
      const nav = await page.$('.top-nav, nav, header');
      if (!nav) throw new Error('No navigation bar');
    }, page);

    await test('2.2 Threads link exists and works', async () => {
      const link = await page.$('a[href*="dashboard"], .top-nav__link--active');
      if (!link) throw new Error('No Threads link');
    }, page);

    await test('2.3 Decisions link exists', async () => {
      const link = await page.$('a[href*="decisions"]');
      if (!link) throw new Error('No Decisions link');
    }, page);

    await test('2.4 Open/Still Open link exists', async () => {
      const link = await page.$('a[href*="still-open"], a[href*="open"]');
      if (!link) throw new Error('No Open items link');
    }, page);

    await test('2.5 Search link exists', async () => {
      const link = await page.$('a[href*="search"], a[href*="memories"]');
      if (!link) throw new Error('No Search link');
    }, page);

    // ========================================
    // SECTION 3: LAYOUT CONSISTENCY
    // ========================================
    log('SECTION 3: Layout Consistency', 'section');

    await test('3.1 Layout is consistent on first load', async () => {
      // Check that we're in the expected initial state
      const sidebar = await page.$('.threads-sidebar');
      const mainContent = await page.$('.threads-main, .thread-detail');
      if (!sidebar) throw new Error('Sidebar not found');
    }, page);

    await test('3.2 Clicking Threads nav maintains layout', async () => {
      const threadsLink = await page.$('a[href*="dashboard"]:not([href*="?"])');
      if (threadsLink) {
        await threadsLink.click();
        await wait(500);
      }
      // Layout should still be consistent
      const sidebar = await page.$('.threads-sidebar');
      if (!sidebar) throw new Error('Layout changed unexpectedly');
    }, page);

    // ========================================
    // SECTION 4: PASTE FLOW
    // ========================================
    log('SECTION 4: Paste Flow', 'section');

    await test('4.1 Paste input area exists', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input, textarea[placeholder*="Paste"], textarea[placeholder*="paste"]');
      if (!input) throw new Error('Paste input not found');
    }, page);

    await test('4.2 Paste input has helpful placeholder', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input');
      if (!input) throw new Error('No input');
      const placeholder = await page.evaluate((el) => el.placeholder, input);
      if (!placeholder || placeholder.length < 10) {
        throw new Error(`Placeholder not helpful: "${placeholder}"`);
      }
    }, page);

    await test('4.3 Can type in paste area', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input');
      if (!input) throw new Error('No input');
      await input.click();
      await input.type('Test input', { delay: 50 });
      const value = await page.evaluate((el) => el.value, input);
      if (!value.includes('Test')) throw new Error('Input not accepting text');
      // Clear it
      await page.evaluate((el) => (el.value = ''), input);
    }, page);

    await test('4.4 Organize/Save button exists', async () => {
      const btn = await page.$('.paste-bar__btn, button[type="submit"], .organize-btn, button:has-text("Organize"), button:has-text("Save")');
      if (!btn) throw new Error('No organize/save button');
    }, page);

    // ========================================
    // SECTION 5: THREAD LIST
    // ========================================
    log('SECTION 5: Thread List', 'section');

    await test('5.1 Thread list container exists', async () => {
      const list = await page.$('#threadsList, .threads-list, .thread-list');
      if (!list) throw new Error('No thread list container');
    }, page);

    await test('5.2 Threads or empty state is shown', async () => {
      await wait(1000); // Wait for data load
      const threads = await page.$$('.thread-item');
      const emptyState = await page.$('.empty-state, .thread-detail__empty, [class*="empty"]');
      if (threads.length === 0 && !emptyState) {
        throw new Error('Neither threads nor empty state shown');
      }
      log(`Found ${threads.length} threads`, 'info');
    }, page);

    await test('5.3 Thread items are clickable', async () => {
      const threads = await page.$$('.thread-item');
      if (threads.length > 0) {
        const isClickable = await page.evaluate((el) => {
          return el.onclick !== null || el.getAttribute('role') === 'button' || el.hasAttribute('tabindex');
        }, threads[0]);
        if (!isClickable) throw new Error('Thread items not clickable');
      }
    }, page);

    await test('5.4 Thread items show platform source', async () => {
      const threads = await page.$$('.thread-item');
      if (threads.length > 0) {
        const html = await page.evaluate((el) => el.innerHTML, threads[0]);
        const hasSource = html.includes('ChatGPT') || html.includes('Claude') ||
                         html.includes('Pasted') || html.includes('from ');
        if (!hasSource) {
          log('Thread source label might be missing', 'warn');
        }
      }
    }, page);

    await test('5.5 Thread items show preview text (not "No preview")', async () => {
      const threads = await page.$$('.thread-item');
      if (threads.length > 0) {
        const html = await page.evaluate((el) => el.innerHTML, threads[0]);
        if (html.includes('No preview')) {
          throw new Error('Thread showing "No preview" - should have actual content');
        }
      }
    }, page);

    // ========================================
    // SECTION 6: THREAD DETAIL VIEW
    // ========================================
    log('SECTION 6: Thread Detail View', 'section');

    let threadClicked = false;
    await test('6.1 Clicking thread opens detail view', async () => {
      const threads = await page.$$('.thread-item');
      if (threads.length > 0) {
        await threads[0].click();
        await wait(800);
        const detail = await page.$('#threadContent, .thread-detail, .thread-content');
        const isVisible = detail && await page.evaluate((el) => {
          const style = window.getComputedStyle(el);
          return style.display !== 'none' && style.visibility !== 'hidden';
        }, detail);
        if (!isVisible) throw new Error('Thread detail not visible after click');
        threadClicked = true;
      } else {
        results.skipped.push('6.1 - No threads to click');
      }
    }, page);

    if (threadClicked) {
      await test('6.2 Thread title is displayed', async () => {
        const title = await page.$('#threadTitle, .thread-header__title');
        if (!title) throw new Error('No thread title');
        const text = await page.evaluate((el) => el.textContent, title);
        if (!text || text.length < 2) throw new Error('Thread title empty');
      }, page);

      await test('6.3 Copy Context button exists', async () => {
        const btn = await page.$('button:has-text("Copy"), .thread-header__btn, .hero-continue-btn');
        if (!btn) throw new Error('No copy context button');
      }, page);

      await test('6.4 "Things still to do" section exists', async () => {
        const section = await page.$('#openSection, [class*="open-section"]');
        // It's OK if hidden when empty, but should exist
      }, page);

      await test('6.5 "What you figured out" section exists', async () => {
        const section = await page.$('#decisionsSection, [class*="decision-section"]');
        // It's OK if hidden when empty, but should exist
      }, page);

      await test('6.6 Edit button exists on items', async () => {
        const editBtns = await page.$$('.open-item__edit, .decision-item__edit, button:has-text("Edit")');
        log(`Found ${editBtns.length} edit buttons`, 'info');
      }, page);
    }

    // ========================================
    // SECTION 7: COPY FUNCTIONALITY
    // ========================================
    log('SECTION 7: Copy Functionality', 'section');

    if (threadClicked) {
      await test('7.1 Copy Context button is clickable', async () => {
        const btn = await page.$('.hero-continue-btn, .thread-header__btn--primary, button:has-text("Copy")');
        if (!btn) throw new Error('No copy button');
        const isDisabled = await page.evaluate((el) => el.disabled, btn);
        if (isDisabled) throw new Error('Copy button is disabled');
      }, page);

      await test('7.2 Copy shows feedback (toast)', async () => {
        const btn = await page.$('.hero-continue-btn, .thread-header__btn--primary');
        if (btn) {
          await btn.click();
          await wait(500);
          // Check for toast
          const toast = await page.$('.haiven-toast, .toast, [role="alert"], [class*="toast"]');
          if (!toast) {
            log('No toast feedback visible after copy', 'warn');
          }
        }
      }, page);
    }

    // ========================================
    // SECTION 8: MOBILE RESPONSIVENESS
    // ========================================
    log('SECTION 8: Mobile Responsiveness', 'section');

    await test('8.1 Mobile viewport renders correctly', async () => {
      await page.setViewport({ width: 375, height: 667 });
      await wait(500);
      const content = await page.$('.threads-dashboard, .dashboard, main');
      if (!content) throw new Error('Content not visible on mobile');
    }, page);

    await test('8.2 Paste input works on mobile', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input');
      if (!input) throw new Error('Paste input not visible on mobile');
    }, page);

    await test('8.3 Navigation adapts to mobile', async () => {
      const nav = await page.$('.top-nav, nav');
      if (!nav) throw new Error('Navigation missing on mobile');
    }, page);

    // Reset viewport
    await page.setViewport({ width: 1400, height: 900 });

    // ========================================
    // SECTION 9: LANGUAGE & COPY CHECK
    // ========================================
    log('SECTION 9: Language & Copy Quality', 'section');

    await test('9.1 No technical jargon visible', async () => {
      const html = await page.content();
      const jargon = ['decompose', 'decomposition', 'thread_id', 'api', 'endpoint', 'Manual'];
      const found = jargon.filter((j) => html.toLowerCase().includes(j.toLowerCase()));
      if (found.length > 0) {
        log(`Technical jargon found: ${found.join(', ')}`, 'warn');
      }
    }, page);

    await test('9.2 No cringe motivational copy', async () => {
      const html = await page.content();
      const cringe = ['every choice tells', 'your journey', 'step forward', 'moments of clarity'];
      const found = cringe.filter((c) => html.toLowerCase().includes(c.toLowerCase()));
      if (found.length > 0) {
        throw new Error(`Cringe copy found: ${found.join(', ')}`);
      }
    }, page);

    await test('9.3 Platform labels use "from X" format', async () => {
      const threads = await page.$$('.thread-item');
      if (threads.length > 0) {
        const html = await page.evaluate((el) => el.innerHTML, threads[0]);
        if (html.includes('Manual') && !html.includes('Pasted')) {
          throw new Error('Still using "Manual" instead of "Pasted"');
        }
      }
    }, page);

    // ========================================
    // SECTION 10: PERFORMANCE
    // ========================================
    log('SECTION 10: Performance', 'section');

    await test('10.1 Page load time under 5 seconds', async () => {
      const start = Date.now();
      await page.goto(DASHBOARD_URL, { waitUntil: 'networkidle2' });
      const loadTime = Date.now() - start;
      log(`Page load time: ${loadTime}ms`, 'info');
      if (loadTime > 5000) {
        throw new Error(`Page took ${loadTime}ms to load`);
      }
    }, page);

    await test('10.2 No memory leaks (basic check)', async () => {
      const metrics = await page.metrics();
      log(`JS Heap: ${Math.round(metrics.JSHeapUsedSize / 1024 / 1024)}MB`, 'info');
      if (metrics.JSHeapUsedSize > 100 * 1024 * 1024) {
        throw new Error('Memory usage too high');
      }
    }, page);

  } catch (err) {
    log(`Test suite error: ${err.message}`, 'fail');
  }

  // ========================================
  // SUMMARY
  // ========================================
  console.log('\n' + '='.repeat(60));
  log('TEST RESULTS SUMMARY', 'section');
  console.log('='.repeat(60));

  console.log(`\n${results.passed.length} passed`);
  console.log(`${results.failed.length} failed`);
  console.log(`${results.skipped.length} skipped`);

  const total = results.passed.length + results.failed.length;
  const score = Math.round((results.passed.length / total) * 100);
  console.log(`\nUX Score: ${score}%`);

  if (results.failed.length > 0) {
    console.log('\n\x1b[31mFailed Tests:\x1b[0m');
    results.failed.forEach((f) => {
      console.log(`  ✗ ${f.name}`);
      console.log(`    → ${f.error}`);
    });
  }

  if (results.skipped.length > 0) {
    console.log('\n\x1b[33mSkipped Tests:\x1b[0m');
    results.skipped.forEach((s) => console.log(`  - ${s}`));
  }

  console.log('\n' + '='.repeat(60));

  if (HEADED) {
    log('Browser staying open. Press Ctrl+C to exit.', 'info');
    await new Promise(() => {});
  } else {
    await browser.close();
  }

  process.exit(results.failed.length > 0 ? 1 : 0);
}

// Run
runTests().catch((err) => {
  console.error('Fatal error:', err);
  process.exit(1);
});
