#!/usr/bin/env node
/**
 * Haiven Dashboard Test
 *
 * Tests the dashboard from an average user's perspective.
 *
 * Usage:
 *   node test-dashboard.js --headed    # Watch it run
 *   node test-dashboard.js --local     # Test localhost:3001
 */

import puppeteer from 'puppeteer';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

const args = process.argv.slice(2);
const HEADED = args.includes('--headed');
const LOCAL = args.includes('--local');

const BASE_URL = LOCAL ? 'http://localhost:3001' : 'https://www.safehaiven.com';
const DASHBOARD_URL = `${BASE_URL}/dashboard.html`;
const EXTENSION_PATH = path.resolve(__dirname);

// Sample chat to paste
const SAMPLE_CHAT = [
  'User: I\'m thinking about launching a newsletter for my coaching business. What platform should I use?',
  '',
  'Assistant: Great question! For a coaching newsletter, I\'d recommend considering:',
  '',
  '1. **Substack** - Best for starting out, free tier, built-in audience discovery',
  '2. **ConvertKit** - More control, better automation, coaching-focused features',
  '3. **Beehiiv** - Modern, great analytics, growing fast',
  '',
  'For your coaching business specifically, I\'d lean toward ConvertKit because:',
  '- You can create automated sequences for new subscribers',
  '- Tag subscribers by interest (mindset, productivity, leadership)',
  '- Sell digital products directly',
  '',
  'User: I like ConvertKit. What should my first email be about?',
  '',
  'Assistant: Your first email should do three things:',
  '',
  '1. **Welcome them** - Thank them for subscribing',
  '2. **Set expectations** - What will you send and how often',
  '3. **Quick win** - Give them one actionable tip they can use today',
  '',
  'Decision: Start with ConvertKit for the newsletter.',
  'To-do: Write the welcome email sequence this week.',
].join('\n');

// Test results
const results = {
  passed: [],
  failed: []
};

function log(msg, type = 'info') {
  const prefix = {
    info: '\x1b[36m[INFO]\x1b[0m',
    pass: '\x1b[32m[PASS]\x1b[0m',
    fail: '\x1b[31m[FAIL]\x1b[0m',
    warn: '\x1b[33m[WARN]\x1b[0m'
  };
  console.log(`${prefix[type] || prefix.info} ${msg}`);
}

async function test(name, fn) {
  try {
    await fn();
    results.passed.push(name);
    log(`${name}`, 'pass');
  } catch (err) {
    results.failed.push({ name, error: err.message });
    log(`${name}: ${err.message}`, 'fail');
  }
}

async function runTests() {
  log('Starting Haiven Dashboard Tests');
  log(`URL: ${DASHBOARD_URL}`);
  log(`Mode: ${HEADED ? 'Headed (visible)' : 'Headless'}`);
  console.log('');

  const browser = await puppeteer.launch({
    headless: !HEADED,
    args: [
      `--disable-extensions-except=${EXTENSION_PATH}`,
      `--load-extension=${EXTENSION_PATH}`,
      '--no-sandbox'
    ],
    defaultViewport: { width: 1400, height: 900 }
  });

  const page = await browser.newPage();

  // Listen for console messages
  page.on('console', msg => {
    if (msg.type() === 'error') {
      log(`Console error: ${msg.text()}`, 'warn');
    }
  });

  try {
    // TEST 1: Dashboard loads
    await test('Dashboard loads', async () => {
      await page.goto(DASHBOARD_URL, { waitUntil: 'networkidle2', timeout: 30000 });
      const title = await page.title();
      if (!title.toLowerCase().includes('haiven')) {
        throw new Error(`Unexpected title: ${title}`);
      }
    });

    // TEST 2: Empty state shows (for new user)
    await test('Empty state is visible', async () => {
      await page.waitForSelector('.empty-state__hook-title, .thread-item', { timeout: 10000 });
      const emptyState = await page.$('.empty-state__hook-title');
      const hasThreads = await page.$('.thread-item');
      if (!emptyState && !hasThreads) {
        throw new Error('Neither empty state nor threads visible');
      }
    });

    // TEST 3: Paste input exists
    await test('Paste input exists', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input, [placeholder*="Paste"]');
      if (!input) {
        throw new Error('Paste input not found');
      }
    });

    // TEST 4: Can paste a conversation
    await test('Can paste conversation', async () => {
      const input = await page.$('#pasteInput, .paste-bar__input');
      if (!input) throw new Error('No paste input');

      await input.click();
      await input.type(SAMPLE_CHAT, { delay: 5 });

      const value = await page.$eval('#pasteInput, .paste-bar__input', el => el.value || el.textContent);
      if (!value || value.length < 50) {
        throw new Error('Paste content not entered');
      }
    });

    // TEST 5: Submit button works
    await test('Submit/Save button exists', async () => {
      const btn = await page.$('.paste-bar__btn, button[type="submit"], .paste-bar__submit');
      if (!btn) {
        throw new Error('Submit button not found');
      }
    });

    // TEST 6: Navigation exists
    await test('Navigation shows key sections', async () => {
      const nav = await page.$('.top-nav, nav');
      if (!nav) throw new Error('No navigation found');

      const html = await page.content();
      const hasDecisions = html.includes('Decisions') || html.includes('decisions');
      const hasStillOpen = html.includes('Still Open') || html.includes('still-open');

      if (!hasDecisions && !hasStillOpen) {
        throw new Error('Key nav items missing');
      }
    });

    // TEST 7: Mobile responsive
    await test('Mobile view works', async () => {
      await page.setViewport({ width: 375, height: 667 });
      await page.waitForTimeout(500);

      const pasteInput = await page.$('#pasteInput, .paste-bar__input');
      if (!pasteInput) {
        throw new Error('Paste input not visible on mobile');
      }

      // Reset viewport
      await page.setViewport({ width: 1400, height: 900 });
    });

    // TEST 8: Welcome greeting appears
    await test('Welcome greeting shows', async () => {
      const greeting = await page.$('#welcomeGreeting, .welcome-banner__greeting');
      if (greeting) {
        const text = await page.$eval('#welcomeGreeting, .welcome-banner__greeting', el => el.textContent);
        if (!text.includes('Good') && !text.includes('Welcome')) {
          throw new Error(`Unexpected greeting: ${text}`);
        }
      }
      // Pass even if no greeting (might be logged out state)
    });

  } catch (err) {
    log(`Test suite error: ${err.message}`, 'fail');
  }

  // Summary
  console.log('\n' + '='.repeat(50));
  log(`Results: ${results.passed.length} passed, ${results.failed.length} failed`);

  if (results.failed.length > 0) {
    console.log('\nFailed tests:');
    results.failed.forEach(f => {
      console.log(`  - ${f.name}: ${f.error}`);
    });
  }

  if (HEADED) {
    log('Browser staying open for inspection. Press Ctrl+C to exit.');
    await new Promise(() => {}); // Keep open
  } else {
    await browser.close();
  }

  process.exit(results.failed.length > 0 ? 1 : 0);
}

runTests().catch(err => {
  console.error('Fatal error:', err);
  process.exit(1);
});
