/* global browser, chrome */
// Browser compatibility layer (use window to share across content scripts)
window.browserAPI = window.browserAPI || (typeof browser !== 'undefined' ? browser : chrome);

// Auth helper - Get headers with API key or userId as fallback
window.getAuthHeaders =
  window.getAuthHeaders ||
  async function () {
    const { apiKey, userId } = await window.browserAPI.storage.local.get(['apiKey', 'userId']);
    const headers = {
      'Content-Type': 'application/json'
    };
    // Use API key if available, otherwise use userId as Bearer token
    if (apiKey) {
      headers['Authorization'] = `Bearer ${apiKey}`;
    } else if (userId) {
      headers['Authorization'] = `Bearer ${userId}`;
    }
    return headers;
  };

/**
 * Haiven Cross-Channel Sync
 *
 * Syncs memories across Extension ↔ MCP ↔ Custom GPT ↔ Dashboard
 * Uses Supabase Realtime (no OpenAI API costs)
 *
 * @version 1.0.0
 */

class HaivenSync {
  constructor() {
    // SECURITY: Get Supabase config from storage (set via settings page)
    // DO NOT hardcode credentials in extension code
    this.supabaseUrl = null;
    this.supabaseAnonKey = null;
    this.channel = null;
    this.userId = null;
    this.listeners = new Map();
    this.connectionAttempts = 0;

    this.init();
  }

  async init() {
    // Get config from storage (set via settings page or API response)
    const { userId, supabaseUrl, supabaseAnonKey } = await window.browserAPI.storage.local.get([
      'userId',
      'supabaseUrl',
      'supabaseAnonKey'
    ]);

    if (!userId) {
      console.warn('[Haiven Sync] No userId found, sync disabled');
      return;
    }

    this.userId = userId;

    // SECURITY: Only use Supabase config from storage, never hardcoded
    if (supabaseUrl && supabaseAnonKey) {
      this.supabaseUrl = supabaseUrl;
      this.supabaseAnonKey = supabaseAnonKey;
      // Connect to Supabase Realtime
      this.connectRealtime();
      console.log('[Haiven Sync] Initialized with realtime for user:', userId);
    } else {
      // Realtime sync disabled - will use polling fallback
      console.log('[Haiven Sync] Realtime config not found, using polling fallback');
      this.usePollingFallback();
    }
  }

  connectRealtime() {
    // Use Supabase Realtime to listen for changes
    // Note: Realtime is optional - core features work without it

    // Stop retrying if we've had too many failures
    if (this.connectionAttempts > 3) {
      console.log('[Haiven Sync] Realtime not available, using polling fallback');
      this.usePollingFallback();
      return;
    }

    this.connectionAttempts = (this.connectionAttempts || 0) + 1;

    // Create WebSocket connection to Supabase (wrapped to suppress errors)
    // Note: API key sent via message after connection, not in URL (security best practice)
    const wsUrl = `${this.supabaseUrl.replace('https', 'wss')}/realtime/v1/websocket?vsn=1.0.0`;

    try {
      this.ws = new WebSocket(wsUrl);
    } catch (_e) {
      // Silent WebSocket creation failure
      return;
    }

    this.ws.onopen = () => {
      console.log('[Haiven Sync] WebSocket connected');
      this.connectionAttempts = 0; // Reset on successful connection

      // Authenticate via message (not URL params)
      const authMessage = {
        topic: 'realtime:system',
        event: 'access_token',
        payload: { access_token: this.supabaseAnonKey },
        ref: '0'
      };
      this.ws.send(JSON.stringify(authMessage));

      // Join channel for this user
      const joinMessage = {
        topic: `realtime:public:memories:user_id=eq.${this.userId}`,
        event: 'phx_join',
        payload: {},
        ref: '1'
      };

      this.ws.send(JSON.stringify(joinMessage));

      // Send heartbeat every 30s to keep connection alive
      this.heartbeatInterval = setInterval(() => {
        if (this.ws && this.ws.readyState === WebSocket.OPEN) {
          this.ws.send(
            JSON.stringify({
              topic: 'phoenix',
              event: 'heartbeat',
              payload: {},
              ref: Date.now().toString()
            })
          );
        }
      }, 30000);
    };

    this.ws.onmessage = (event) => {
      const data = JSON.parse(event.data);

      // Handle realtime events
      if (data.event === 'INSERT' || data.event === 'UPDATE') {
        this.handleMemoryUpdate(data.payload);
      }
    };

    this.ws.onerror = (_error) => {
      // Silent failure - realtime is optional (suppress console warnings)
    };

    this.ws.onclose = () => {
      // Clear heartbeat
      if (this.heartbeatInterval) {
        clearInterval(this.heartbeatInterval);
      }

      // Only retry a few times (silent retries)
      if (this.connectionAttempts <= 3) {
        // Silent reconnect attempt
        setTimeout(() => this.connectRealtime(), 5000);
      }
    };
  }

  usePollingFallback() {
    // Fallback to occasional polling instead of realtime
    // This is less efficient but works without Realtime enabled
    console.log('[Haiven Sync] Using polling mode (check every 60s)');

    setInterval(() => {
      // Silently check for updates via API
      this.triggerSync();
    }, 60000); // Check every minute
  }

  handleMemoryUpdate(payload) {
    console.log('[Haiven Sync] Memory updated:', payload);

    // Get the memory data
    const memory = payload.new || payload.record;

    // Notify all listeners
    this.listeners.forEach((callback, event) => {
      if (event === 'memory:created' || event === 'memory:updated') {
        callback(memory);
      }
    });

    // Update local cache
    this.updateLocalCache(memory);

    // Show notification
    this.showSyncNotification(memory);
  }

  async updateLocalCache(memory) {
    try {
      // Get current cached memories
      const { cachedMemories = [] } = await window.browserAPI.storage.local.get(['cachedMemories']);

      // Check if memory already exists
      const existingIndex = cachedMemories.findIndex((m) => m.id === memory.id);

      if (existingIndex >= 0) {
        // Update existing
        cachedMemories[existingIndex] = memory;
      } else {
        // Add new (keep only last 100 in cache)
        cachedMemories.unshift(memory);
        if (cachedMemories.length > 100) {
          cachedMemories.pop();
        }
      }

      // Save back to storage
      await window.browserAPI.storage.local.set({ cachedMemories });

      console.log('[Haiven Sync] Local cache updated');
    } catch (error) {
      console.error('[Haiven Sync] Cache update failed:', error);
    }
  }

  showSyncNotification(memory) {
    // Get source platform
    const source = memory.metadata?.platform || 'Unknown';

    // Only show notification if from different platform than current
    const currentPlatform = this.detectPlatform();

    if (source !== currentPlatform && source !== 'Unknown') {
      // Create subtle toast notification
      this.createToast(`New memory from ${source}`, `${memory.content?.substring(0, 60)}...`);
    }
  }

  // SECURITY: Escape HTML to prevent XSS
  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  createToast(title, message) {
    // Create toast element
    const toast = document.createElement('div');
    toast.id = 'haiven-sync-toast';
    toast.style.cssText = `
      position: fixed;
      bottom: 24px;
      left: 24px;
      background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
      border: 1px solid rgba(0, 212, 255, 0.3);
      border-radius: 12px;
      padding: 16px;
      max-width: 350px;
      box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
      z-index: 999999;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
      color: #ffffff;
      animation: haiven-toast-slide-in 0.3s ease-out;
    `;

    // SECURITY: Build DOM safely without innerHTML for user content
    // Create style element for animations
    const style = document.createElement('style');
    style.textContent = `
      @keyframes haiven-toast-slide-in {
        from { opacity: 0; transform: translateX(-100%); }
        to { opacity: 1; transform: translateX(0); }
      }
      @keyframes haiven-toast-slide-out {
        from { opacity: 1; transform: translateX(0); }
        to { opacity: 0; transform: translateX(-100%); }
      }
    `;
    toast.appendChild(style);

    // Create content container
    const container = document.createElement('div');
    container.style.cssText = 'display: flex; align-items: start; gap: 12px;';

    const icon = document.createElement('div');
    icon.style.fontSize = '24px';
    icon.textContent = '🔄';
    container.appendChild(icon);

    const content = document.createElement('div');
    content.style.flex = '1';

    const titleEl = document.createElement('div');
    titleEl.style.cssText =
      'font-size: 14px; font-weight: 700; margin-bottom: 4px; color: #00d4ff;';
    titleEl.textContent = title; // SAFE: textContent escapes HTML

    const messageEl = document.createElement('div');
    messageEl.style.cssText = 'font-size: 12px; color: rgba(255, 255, 255, 0.7);';
    messageEl.textContent = message; // SAFE: textContent escapes HTML

    content.appendChild(titleEl);
    content.appendChild(messageEl);
    container.appendChild(content);
    toast.appendChild(container);

    document.body.appendChild(toast);

    // Auto-remove after 4 seconds
    setTimeout(() => {
      toast.style.animation = 'haiven-toast-slide-out 0.3s ease-out';
      setTimeout(() => toast.remove(), 300);
    }, 4000);
  }

  // Event listeners
  on(event, callback) {
    this.listeners.set(event, callback);
  }

  off(event) {
    this.listeners.delete(event);
  }

  // Trigger sync manually
  async triggerSync() {
    try {
      // Ensure we have userId
      if (!this.userId) {
        const { userId } = await window.browserAPI.storage.local.get(['userId']);
        if (!userId) {
          console.warn('[Haiven Sync] No userId for sync');
          return;
        }
        this.userId = userId;
      }

      const headers = await window.getAuthHeaders();
      const response = await fetch('https://api.safehaiven.com/api/memory/sync', {
        method: 'POST',
        headers: headers,
        body: JSON.stringify({
          userId: this.userId,
          source: 'extension'
        })
      });

      if (response.ok) {
        const data = await response.json();
        console.log('[Haiven Sync] Manual sync triggered:', data.synced, 'memories');
      } else {
        console.error('[Haiven Sync] Sync response error:', response.status);
      }
    } catch (error) {
      console.error('[Haiven Sync] Manual sync failed:', error.message || error);
    }
  }

  // Get cached memories
  async getCachedMemories() {
    const { cachedMemories = [] } = await window.browserAPI.storage.local.get(['cachedMemories']);
    return cachedMemories;
  }

  // Detect current platform
  detectPlatform() {
    const url = window.location.hostname;
    if (url.includes('chatgpt.com') || url.includes('chat.openai.com')) return 'ChatGPT';
    if (url.includes('claude.ai')) return 'Claude';
    if (url.includes('grok')) return 'Grok';
    if (url.includes('perplexity.ai')) return 'Perplexity';
    if (url.includes('gemini.google.com')) return 'Gemini';
    return 'Unknown';
  }

  // Cleanup
  disconnect() {
    if (this.heartbeatInterval) {
      clearInterval(this.heartbeatInterval);
    }
    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }
    this.listeners.clear();
  }
}

// Initialize sync
const haivenSync = new HaivenSync();

// Export for use in other scripts
if (typeof window !== 'undefined') {
  window.haivenSync = haivenSync;
}
