/* global browser, chrome */
// Browser compatibility layer
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
// Settings page logic
document.addEventListener('DOMContentLoaded', async () => {
  console.log('Settings page loaded');

  // Load existing settings
  const settings = await browserAPI.storage.local.get([
    'apiEndpoint',
    'userId',
    'apiKey',
    'dashboardUrl',
    'autoCapture',
    'showSaveOnImages',
    'showSaveOnCode',
    'showSaveOnArtifacts'
  ]);
  console.log('Loaded settings:', settings);

  const apiEndpointEl = document.getElementById('apiEndpoint');
  const userIdEl = document.getElementById('userId');
  const apiKeyEl = document.getElementById('apiKey');
  const dashboardUrlEl = document.getElementById('dashboardUrl');
  const autoCaptureEl = document.getElementById('autoCapture');

  if (!apiEndpointEl || !userIdEl || !dashboardUrlEl || !autoCaptureEl) {
    console.error('Missing settings elements:', {
      apiEndpoint: !!apiEndpointEl,
      userId: !!userIdEl,
      apiKey: !!apiKeyEl,
      dashboardUrl: !!dashboardUrlEl,
      autoCapture: !!autoCaptureEl
    });
    return;
  }

  apiEndpointEl.value = settings.apiEndpoint || 'https://api.safehaiven.com';
  userIdEl.value = settings.userId || '';
  if (apiKeyEl) apiKeyEl.value = settings.apiKey || '';
  dashboardUrlEl.value = settings.dashboardUrl || 'https://www.safehaiven.com';
  autoCaptureEl.checked = settings.autoCapture !== false; // Default: true

  // Save Generated Content settings (default: true)
  const showSaveOnImagesEl = document.getElementById('showSaveOnImages');
  const showSaveOnCodeEl = document.getElementById('showSaveOnCode');
  const showSaveOnArtifactsEl = document.getElementById('showSaveOnArtifacts');

  if (showSaveOnImagesEl) showSaveOnImagesEl.checked = settings.showSaveOnImages !== false;
  if (showSaveOnCodeEl) showSaveOnCodeEl.checked = settings.showSaveOnCode !== false;
  if (showSaveOnArtifactsEl) showSaveOnArtifactsEl.checked = settings.showSaveOnArtifacts !== false;

  console.log('Settings populated successfully');
});

// Preset buttons
document.addEventListener('DOMContentLoaded', () => {
  const presetButtons = document.querySelectorAll('.preset-btn');
  console.log('Found preset buttons:', presetButtons.length);

  presetButtons.forEach((btn) => {
    btn.addEventListener('click', () => {
      const apiEndpointEl = document.getElementById('apiEndpoint');
      if (apiEndpointEl) {
        apiEndpointEl.value = btn.dataset.url;
        console.log('Set API endpoint to:', btn.dataset.url);
      }
    });
  });
});

// Save button
document.addEventListener('DOMContentLoaded', () => {
  const saveBtn = document.getElementById('saveBtn');
  if (!saveBtn) {
    console.error('Save button not found');
    return;
  }

  saveBtn.addEventListener('click', async (e) => {
    e.preventDefault(); // Prevent any default behavior
    console.log('Save button clicked');

    const apiEndpointEl = document.getElementById('apiEndpoint');
    const userIdEl = document.getElementById('userId');
    const apiKeyEl = document.getElementById('apiKey');
    const dashboardUrlEl = document.getElementById('dashboardUrl');
    const autoCaptureEl = document.getElementById('autoCapture');

    if (!apiEndpointEl || !userIdEl || !dashboardUrlEl || !autoCaptureEl) {
      console.error('Missing form elements');
      showStatus('❌ Form elements not found', 'error');
      return;
    }

    const apiEndpoint = apiEndpointEl.value.trim();
    const userId = userIdEl.value.trim();
    const apiKey = apiKeyEl ? apiKeyEl.value.trim() : '';
    const dashboardUrl = dashboardUrlEl.value.trim();
    const autoCapture = autoCaptureEl.checked;

    // Save Generated Content settings
    const showSaveOnImagesEl = document.getElementById('showSaveOnImages');
    const showSaveOnCodeEl = document.getElementById('showSaveOnCode');
    const showSaveOnArtifactsEl = document.getElementById('showSaveOnArtifacts');
    const showSaveOnImages = showSaveOnImagesEl ? showSaveOnImagesEl.checked : true;
    const showSaveOnCode = showSaveOnCodeEl ? showSaveOnCodeEl.checked : true;
    const showSaveOnArtifacts = showSaveOnArtifactsEl ? showSaveOnArtifactsEl.checked : true;

    if (!apiEndpoint) {
      showStatus('API Endpoint is required', 'error');
      return;
    }

    if (!userId) {
      showStatus('User ID is required', 'error');
      return;
    }

    // Validate User ID format (should be a UUID)
    const uuidRegex = /^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i;
    if (!uuidRegex.test(userId)) {
      showStatus('⚠️ Invalid User ID format (should be a UUID)', 'error');
      return;
    }

    showStatus('⏳ Saving settings...', 'info');

    try {
      // Save settings
      await browserAPI.storage.local.set({
        apiEndpoint,
        userId,
        apiKey,
        dashboardUrl: dashboardUrl || 'https://www.safehaiven.com',
        autoCapture,
        showSaveOnImages,
        showSaveOnCode,
        showSaveOnArtifacts
      });

      showStatus('✅ Settings saved successfully!', 'success');

      // Clear after 2 seconds
      setTimeout(() => {
        const container = document.getElementById('statusContainer');
        if (container) {
          const status = container.querySelector('.status');
          if (status) {
            status.style.display = 'none';
          }
        }
      }, 2000);
    } catch (error) {
      console.error('Save error:', error);
      showStatus('❌ Failed to save settings', 'error');
    }
  });
});

function showStatus(message, type) {
  const container = document.getElementById('statusContainer');
  if (!container) {
    console.error('statusContainer not found');
    return;
  }

  // Create or update status div
  let status = container.querySelector('.status');
  if (!status) {
    status = document.createElement('div');
    status.className = 'status';
    container.appendChild(status);
  }

  status.textContent = message;
  status.className = `status ${type}`;
  status.style.display = 'flex';
}

// EMERGENCY: Force save on Alt+S
document.addEventListener('keydown', async (e) => {
  if (e.altKey && e.key === 's') {
    e.preventDefault();
    console.log('🚨 EMERGENCY SAVE TRIGGERED');

    const userId = document.getElementById('userId')?.value?.trim();

    // Require valid userId - no fallback to hardcoded values
    if (!userId) {
      showStatus('❌ Please enter your User ID first', 'error');
      return;
    }

    try {
      await browserAPI.storage.local.set({
        userId: userId,
        apiEndpoint: 'https://api.safehaiven.com',
        dashboardUrl: 'https://www.safehaiven.com',
        autoCapture: true
      });

      console.log('✅ EMERGENCY SAVE SUCCESS!');
      showStatus('✅ Settings saved via emergency shortcut!', 'success');
    } catch (error) {
      console.error('❌ EMERGENCY SAVE FAILED:', error);
      showStatus(`❌ Save failed: ${error.message}`, 'error');
    }
  }
});
