/* global browser, chrome */
// Browser compatibility layer (use window to share across content scripts)
window.browserAPI = window.browserAPI || (typeof browser !== 'undefined' ? browser : chrome);

// Auth helper - Get headers with API key or userId as fallback (use window to share)
window.getAuthHeaders =
  window.getAuthHeaders ||
  async function () {
    const { apiKey, userId } = await window.browserAPI.storage.local.get(['apiKey', 'userId']);
    const headers = {
      'Content-Type': 'application/json'
    };
    // Use API key if available, otherwise use userId as Bearer token
    if (apiKey) {
      headers['Authorization'] = `Bearer ${apiKey}`;
    } else if (userId) {
      headers['Authorization'] = `Bearer ${userId}`;
    }
    return headers;
  };

/**
 * Haiven Predictive Memory Overlay
 *
 * THE MAGIC: Shows relevant memories while user types in ChatGPT/Claude/Grok
 *
 * This is the "holy shit" moment - cross-LLM memory recall in real-time
 * Non-intrusive, beautiful, fast (<500ms response)
 *
 * @version 2.1.0 - Added mini-dashboard with floating button
 *
 * Features:
 * - Predictive memory overlay while typing
 * - Mini dashboard button (bottom-left) with task badge
 * - Shows: stats, open tasks, active threads, recent memories
 * - Keyboard shortcut: Ctrl+Shift+H to toggle dashboard
 */

class PredictiveMemoryOverlay {
  constructor() {
    this.overlay = null;
    this.searchTimeout = null;
    this.lastQuery = '';
    this.lastInjectedQuery = ''; // Track what we injected for - don't show again for same context
    this.lastInjectionTime = 0; // Timestamp of last injection
    this.sessionInjected = false; // Flag: has user injected in this session?
    this.currentMemories = [];
    this.isVisible = false;
    this.isLoading = false; // Track loading state
    this.selectedIndex = -1; // Track keyboard selection
    this.recentlyInjectedIds = new Set(); // Track recently injected memory IDs
    this.toastContainer = null; // Toast notification container
    this.toastQueue = []; // Queue for managing multiple toasts
    this.dashboardButton = null; // Quick dashboard toggle button
    this.dashboardData = null; // Cached dashboard data
    this.settings = {
      minChars: 15, // Min characters before searching
      debounceMs: 500, // Wait for user to stop typing
      maxResults: 5, // Show top 5 memories (increased from 3)
      minRelevance: 0.25, // Minimum 25% relevance (lowered from 40%)
      position: 'bottom-right', // Where to show overlay
      minQueryChange: 0.3, // Require 30% query change to show overlay again after injection
      recentInjectionTimeout: 5000, // Clear "recently injected" indicator after 5 seconds
      sessionCooldownMs: 120000, // 2 minutes cooldown after injection before showing again
      toastDuration: 3000, // Default toast duration
      maxToasts: 4 // Maximum visible toasts
    };

    // User learning - track behavior to improve recall
    this.userBehavior = {
      usedMemoryIds: [], // Memories user actually used
      dismissedMemoryIds: [], // Memories user dismissed
      platformStats: {}, // Usage per platform
      topicAffinities: {} // Topics user engages with
    };

    // Load saved behavior from storage
    this.loadUserBehavior();

    this.init();
  }

  init() {
    // Create overlay DOM element (hidden by default)
    this.createOverlay();

    // Create toast container
    this.createToastContainer();

    // Create floating dashboard button
    this.createDashboardButton();

    // Listen for typing in text areas
    this.attachInputListeners();

    // Listen for keyboard shortcuts
    this.attachKeyboardShortcuts();

    // Prefetch dashboard data in background
    this.prefetchDashboardData();

    console.log('[Haiven Predictive] Initialized with mini-dashboard');
  }

  createOverlay() {
    // Create overlay container
    this.overlay = document.createElement('div');
    this.overlay.id = 'haiven-predictive-overlay';
    this.overlay.className = 'haiven-overlay hidden';

    this.overlay.innerHTML = `
      <div class="haiven-overlay-header">
        <span class="haiven-icon">🧠</span>
        <span class="haiven-title">Related memories (<span id="haiven-count">0</span>)</span>
        <button class="haiven-close" title="Dismiss">×</button>
      </div>
      <div class="haiven-overlay-content" id="haiven-memories"></div>
      <div class="haiven-overlay-footer">
        <button class="haiven-btn-secondary" id="haiven-dismiss">Dismiss</button>
        <button class="haiven-btn-primary" id="haiven-add-all">Add all</button>
      </div>
    `;

    // Add styles
    this.injectStyles();

    // Add to page
    document.body.appendChild(this.overlay);

    // Attach event listeners - pass true to track user dismissals
    this.overlay.querySelector('.haiven-close').addEventListener('click', () => this.hide(true));
    this.overlay.querySelector('#haiven-dismiss').addEventListener('click', () => this.hide(true));
    this.overlay
      .querySelector('#haiven-add-all')
      .addEventListener('click', () => this.addAllMemories());

    // Make draggable
    this.makeDraggable();
  }

  injectStyles() {
    if (document.getElementById('haiven-predictive-styles')) return;

    const style = document.createElement('style');
    style.id = 'haiven-predictive-styles';
    style.textContent = `
      /* Haiven Predictive Overlay - Premium Natural Design */
      .haiven-overlay {
        position: fixed;
        bottom: 24px;
        right: 24px;
        width: min(380px, calc(100vw - 48px));
        max-height: min(480px, calc(100vh - 48px));
        background: rgba(17, 17, 23, 0.95);
        backdrop-filter: blur(40px) saturate(180%);
        -webkit-backdrop-filter: blur(40px) saturate(180%);
        border-radius: 16px;
        border: 1px solid rgba(255, 255, 255, 0.06);
        box-shadow:
          0 24px 48px -12px rgba(0, 0, 0, 0.5),
          0 0 0 1px rgba(255, 255, 255, 0.03);
        z-index: 999999;
        font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'Segoe UI', Roboto, sans-serif;
        color: #ffffff;
        overflow: hidden;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        animation: haiven-overlay-entrance 0.4s cubic-bezier(0.4, 0, 0.2, 1);
      }

      /* Subtle border glow */
      .haiven-overlay::before {
        content: '';
        position: absolute;
        inset: -1px;
        background: linear-gradient(135deg, rgba(99, 102, 241, 0.2), rgba(168, 85, 247, 0.2));
        border-radius: 17px;
        z-index: -1;
        opacity: 0;
        transition: opacity 0.3s ease;
      }

      .haiven-overlay:hover::before {
        opacity: 1;
      }

      .haiven-overlay.hidden {
        opacity: 0;
        transform: translateY(30px) scale(0.92);
        pointer-events: none;
      }

      @keyframes haiven-overlay-entrance {
        0% {
          opacity: 0;
          transform: translateY(40px) scale(0.9);
          filter: blur(8px);
        }
        100% {
          opacity: 1;
          transform: translateY(0) scale(1);
          filter: blur(0);
        }
      }

      .haiven-overlay-header {
        display: flex;
        align-items: center;
        gap: 10px;
        padding: 14px 16px;
        border-bottom: 1px solid rgba(255, 255, 255, 0.04);
        cursor: grab;
      }

      .haiven-overlay-header:active {
        cursor: grabbing;
      }

      .haiven-icon {
        font-size: 18px;
        opacity: 0.9;
      }

      .haiven-title {
        flex: 1;
        font-size: 13px;
        font-weight: 500;
        color: rgba(255, 255, 255, 0.7);
      }

      #haiven-count {
        color: rgba(255, 255, 255, 0.9);
        font-weight: 600;
      }

      .haiven-close {
        width: 28px;
        height: 28px;
        border: none;
        background: transparent;
        color: rgba(255, 255, 255, 0.4);
        border-radius: 6px;
        cursor: pointer;
        font-size: 16px;
        line-height: 1;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        justify-content: center;
      }

      .haiven-close:hover {
        background: rgba(255, 255, 255, 0.1);
        color: rgba(255, 255, 255, 0.8);
      }

      .haiven-overlay-content {
        padding: 12px;
        max-height: min(340px, calc(100vh - 180px));
        overflow-y: auto;
      }

      .haiven-overlay-content::-webkit-scrollbar {
        width: 6px;
      }

      .haiven-overlay-content::-webkit-scrollbar-track {
        background: transparent;
      }

      .haiven-overlay-content::-webkit-scrollbar-thumb {
        background: rgba(255, 255, 255, 0.1);
        border-radius: 3px;
      }

      .haiven-memory-card {
        background: rgba(255, 255, 255, 0.03);
        border: 1px solid rgba(255, 255, 255, 0.04);
        border-radius: 12px;
        padding: 14px;
        margin-bottom: 8px;
        cursor: pointer;
        transition: all 0.2s ease;
        position: relative;
      }

      .haiven-memory-card:hover {
        background: rgba(255, 255, 255, 0.06);
        border-color: rgba(99, 102, 241, 0.3);
      }

      .haiven-memory-card:last-child {
        margin-bottom: 0;
      }

      /* Recency indicator - subtle left border */
      .haiven-memory-card.haiven-fresh {
        border-left: 2px solid #22c55e;
      }

      .haiven-memory-card.haiven-recent {
        border-left: 2px solid #6366f1;
      }

      .haiven-memory-card.haiven-moderate {
        border-left: 2px solid #f59e0b;
      }

      .haiven-memory-card.haiven-old {
        border-left: 2px solid rgba(148, 163, 184, 0.3);
      }

      .haiven-memory-card:nth-child(1) { animation: haiven-card-entrance 0.3s ease both; animation-delay: 0.05s; }
      .haiven-memory-card:nth-child(2) { animation: haiven-card-entrance 0.3s ease both; animation-delay: 0.1s; }
      .haiven-memory-card:nth-child(3) { animation: haiven-card-entrance 0.3s ease both; animation-delay: 0.15s; }

      @keyframes haiven-card-entrance {
        from {
          opacity: 0;
          transform: translateY(8px);
        }
        to {
          opacity: 1;
          transform: translateY(0);
        }
      }

      .haiven-memory-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 8px;
      }

      .haiven-memory-source {
        display: flex;
        align-items: center;
        gap: 6px;
      }

      .haiven-platform-icon {
        font-size: 12px;
      }

      .haiven-meta {
        font-size: 11px;
        color: rgba(255, 255, 255, 0.5);
      }

      .haiven-platform {
        font-weight: 500;
        color: rgba(255, 255, 255, 0.7);
      }

      .haiven-confidence {
        display: flex;
        align-items: center;
        gap: 4px;
      }

      .haiven-confidence-text {
        font-size: 11px;
        font-weight: 600;
        color: #6366f1;
      }

      .haiven-memory-content {
        font-size: 13px;
        line-height: 1.5;
        color: rgba(255, 255, 255, 0.9);
        display: -webkit-box;
        -webkit-line-clamp: 2;
        -webkit-box-orient: vertical;
        overflow: hidden;
      }

      .haiven-memory-footer {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-top: 10px;
        padding-top: 10px;
        border-top: 1px solid rgba(255, 255, 255, 0.04);
      }

      .haiven-memory-world {
        font-size: 10px;
        padding: 3px 8px;
        border-radius: 4px;
        background: rgba(99, 102, 241, 0.15);
        color: rgba(255, 255, 255, 0.7);
        font-weight: 500;
      }

      .haiven-add-btn {
        padding: 6px 12px;
        background: rgba(99, 102, 241, 0.9);
        border: none;
        border-radius: 6px;
        color: #ffffff;
        font-size: 12px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
        display: flex;
        align-items: center;
        gap: 6px;
      }

      .haiven-add-btn:hover {
        background: rgba(99, 102, 241, 1);
        transform: translateY(-1px);
      }

      .haiven-add-btn:active {
        transform: translateY(0);
      }

      .haiven-btn-shortcut {
        font-size: 10px;
        padding: 2px 4px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 3px;
      }

      .haiven-overlay-footer {
        display: flex;
        gap: 8px;
        padding: 12px 16px;
        border-top: 1px solid rgba(255, 255, 255, 0.04);
      }

      .haiven-btn-secondary {
        flex: 1;
        padding: 10px 14px;
        background: transparent;
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 8px;
        color: rgba(255, 255, 255, 0.7);
        font-size: 12px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
      }

      .haiven-btn-secondary:hover {
        background: rgba(255, 255, 255, 0.05);
        border-color: rgba(255, 255, 255, 0.15);
      }

      .haiven-btn-primary {
        flex: 1;
        padding: 10px 14px;
        background: rgba(99, 102, 241, 0.9);
        border: none;
        border-radius: 8px;
        color: #ffffff;
        font-size: 12px;
        font-weight: 500;
        cursor: pointer;
        transition: all 0.2s ease;
      }

      .haiven-btn-primary:hover {
        background: rgba(99, 102, 241, 1);
      }

      /* Empty State */
      .haiven-empty-state {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 48px 32px;
        text-align: center;
        animation: haiven-fade-in 0.4s ease-out;
      }

      .haiven-empty-icon {
        font-size: 64px;
        margin-bottom: 16px;
        opacity: 0.4;
        animation: haiven-float 3s ease-in-out infinite;
      }

      @keyframes haiven-float {
        0%, 100% { transform: translateY(0px); }
        50% { transform: translateY(-8px); }
      }

      .haiven-empty-title {
        font-size: 16px;
        font-weight: 700;
        color: rgba(255, 255, 255, 0.9);
        margin-bottom: 8px;
      }

      .haiven-empty-message {
        font-size: 14px;
        line-height: 1.6;
        color: rgba(255, 255, 255, 0.5);
        margin-bottom: 24px;
        max-width: 300px;
      }

      .haiven-empty-cta {
        display: inline-flex;
        align-items: center;
        gap: 8px;
        padding: 10px 20px;
        background: linear-gradient(135deg, #00d4ff 0%, #0099cc 100%);
        border: none;
        border-radius: 8px;
        color: #ffffff;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        text-decoration: none;
        transition: all 0.2s;
      }

      .haiven-empty-cta:hover {
        transform: translateY(-2px);
        box-shadow: 0 8px 24px rgba(0, 212, 255, 0.3);
      }

      .haiven-empty-tips {
        margin-top: 24px;
        padding-top: 24px;
        border-top: 1px solid rgba(255, 255, 255, 0.1);
      }

      .haiven-empty-tips-title {
        font-size: 12px;
        font-weight: 600;
        color: rgba(255, 255, 255, 0.6);
        margin-bottom: 12px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
      }

      .haiven-empty-tip {
        display: flex;
        align-items: flex-start;
        gap: 8px;
        text-align: left;
        margin-bottom: 10px;
        font-size: 13px;
        color: rgba(255, 255, 255, 0.5);
        line-height: 1.5;
      }

      .haiven-empty-tip-icon {
        flex-shrink: 0;
        width: 20px;
        height: 20px;
        display: flex;
        align-items: center;
        justify-content: center;
        background: rgba(0, 212, 255, 0.15);
        border-radius: 4px;
        font-size: 12px;
      }

      /* Loading State */
      .haiven-loading {
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 48px 32px;
        gap: 16px;
      }

      .haiven-spinner {
        width: 40px;
        height: 40px;
        border: 3px solid rgba(255, 255, 255, 0.1);
        border-top-color: #00d4ff;
        border-radius: 50%;
        animation: haiven-spin 0.8s linear infinite;
      }

      @keyframes haiven-spin {
        to { transform: rotate(360deg); }
      }

      .haiven-loading-text {
        font-size: 14px;
        color: rgba(255, 255, 255, 0.6);
        animation: haiven-pulse 1.5s ease-in-out infinite;
      }

      @keyframes haiven-pulse {
        0%, 100% { opacity: 0.6; }
        50% { opacity: 1; }
      }

      /* Keyboard Selection Highlight */
      .haiven-memory-card.haiven-selected {
        background: rgba(99, 102, 241, 0.1);
        border-color: rgba(99, 102, 241, 0.4);
      }

      /* Recently Added Badge */
      .haiven-recently-injected {
        position: absolute;
        top: 10px;
        right: 10px;
        background: rgba(34, 197, 94, 0.2);
        color: #22c55e;
        font-size: 9px;
        font-weight: 600;
        padding: 3px 6px;
        border-radius: 4px;
        text-transform: uppercase;
        letter-spacing: 0.3px;
      }

      /* Memory Preview Tooltip */
      .haiven-memory-preview {
        position: fixed;
        background: rgba(17, 17, 23, 0.98);
        border: 1px solid rgba(255, 255, 255, 0.08);
        border-radius: 12px;
        padding: 16px;
        max-width: 360px;
        box-shadow: 0 20px 40px -10px rgba(0, 0, 0, 0.5);
        z-index: 9999999;
        animation: haiven-preview-in 0.2s ease;
        pointer-events: none;
        backdrop-filter: blur(20px);
      }

      @keyframes haiven-preview-in {
        from {
          opacity: 0;
          transform: scale(0.95);
        }
        to {
          opacity: 1;
          transform: scale(1);
        }
      }

      .haiven-preview-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 10px;
        font-size: 11px;
      }

      .haiven-preview-platform {
        color: #6366f1;
        font-weight: 500;
      }

      .haiven-preview-time {
        color: rgba(255, 255, 255, 0.4);
      }

      .haiven-preview-content {
        font-size: 13px;
        line-height: 1.6;
        color: rgba(255, 255, 255, 0.85);
        max-height: 250px;
        overflow-y: auto;
      }

      .haiven-preview-world {
        margin-top: 10px;
        font-size: 10px;
        color: rgba(255, 255, 255, 0.5);
      }

      /* Button Text */
      .haiven-btn-text {
        display: inline;
      }

      /* Smooth Fade Transitions */
      .haiven-overlay {
        transition: opacity 0.3s cubic-bezier(0.4, 0, 0.2, 1),
                    transform 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      }

      .haiven-overlay.hidden {
        opacity: 0;
        transform: translateY(20px) scale(0.95);
        pointer-events: none;
      }

      .haiven-overlay:not(.hidden) {
        opacity: 1;
        transform: translateY(0) scale(1);
      }
    `;

    document.head.appendChild(style);
  }

  createToastContainer() {
    // Create toast container (fixed position at bottom-left)
    this.toastContainer = document.createElement('div');
    this.toastContainer.id = 'haiven-toast-container';
    this.toastContainer.className = 'haiven-toast-container';
    document.body.appendChild(this.toastContainer);

    // Add toast styles
    this.injectToastStyles();
  }

  injectToastStyles() {
    if (document.getElementById('haiven-toast-styles')) return;

    const style = document.createElement('style');
    style.id = 'haiven-toast-styles';
    style.textContent = `
      /* Toast Container */
      .haiven-toast-container {
        position: fixed;
        bottom: 24px;
        left: 24px;
        z-index: 9999999;
        display: flex;
        flex-direction: column-reverse;
        gap: 10px;
        pointer-events: none;
      }

      /* Toast - Clean, prominent design */
      .haiven-toast {
        min-width: 280px;
        max-width: 360px;
        background: rgba(17, 17, 23, 0.98);
        backdrop-filter: blur(20px);
        border-radius: 10px;
        padding: 14px 16px;
        display: flex;
        align-items: center;
        gap: 12px;
        box-shadow: 0 16px 32px rgba(0, 0, 0, 0.4);
        animation: haiven-toast-in 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        pointer-events: auto;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        position: relative;
        overflow: hidden;
      }

      .haiven-toast.haiven-toast-out {
        animation: haiven-toast-out 0.25s ease forwards;
      }

      @keyframes haiven-toast-in {
        from {
          opacity: 0;
          transform: translateY(10px) scale(0.95);
        }
        to {
          opacity: 1;
          transform: translateY(0) scale(1);
        }
      }

      @keyframes haiven-toast-out {
        to {
          opacity: 0;
          transform: translateY(10px) scale(0.95);
        }
      }

      .haiven-toast-icon {
        flex-shrink: 0;
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        border-radius: 8px;
      }

      .haiven-toast-content {
        flex: 1;
      }

      .haiven-toast-title {
        font-size: 13px;
        font-weight: 600;
        color: #ffffff;
        margin-bottom: 2px;
      }

      .haiven-toast-message {
        font-size: 12px;
        line-height: 1.4;
        color: rgba(255, 255, 255, 0.6);
      }

      .haiven-toast-close {
        flex-shrink: 0;
        width: 24px;
        height: 24px;
        border: none;
        background: transparent;
        color: rgba(255, 255, 255, 0.4);
        border-radius: 4px;
        cursor: pointer;
        font-size: 14px;
        transition: all 0.2s;
        display: flex;
        align-items: center;
        justify-content: center;
      }

      .haiven-toast-close:hover {
        background: rgba(255, 255, 255, 0.1);
        color: rgba(255, 255, 255, 0.8);
      }

      /* Toast Types - Prominent success state */
      .haiven-toast.haiven-toast-success {
        border-left: 3px solid #22c55e;
      }

      .haiven-toast.haiven-toast-success .haiven-toast-icon {
        background: rgba(34, 197, 94, 0.15);
        color: #22c55e;
      }

      .haiven-toast.haiven-toast-error {
        border-left: 3px solid #ef4444;
      }

      .haiven-toast.haiven-toast-error .haiven-toast-icon {
        background: rgba(239, 68, 68, 0.15);
        color: #ef4444;
      }

      .haiven-toast.haiven-toast-warning {
        border-left: 3px solid #f59e0b;
      }

      .haiven-toast.haiven-toast-warning .haiven-toast-icon {
        background: rgba(245, 158, 11, 0.15);
        color: #f59e0b;
      }

      .haiven-toast.haiven-toast-info {
        border-left: 3px solid #6366f1;
      }

      .haiven-toast.haiven-toast-info .haiven-toast-icon {
        background: rgba(99, 102, 241, 0.15);
        color: #6366f1;
      }

      /* Progress Bar */
      .haiven-toast-progress {
        position: absolute;
        bottom: 0;
        left: 0;
        right: 0;
        height: 2px;
        background: rgba(255, 255, 255, 0.1);
        overflow: hidden;
      }

      .haiven-toast-progress-bar {
        height: 100%;
        background: rgba(255, 255, 255, 0.3);
        animation: haiven-toast-progress-anim linear forwards;
      }

      @keyframes haiven-toast-progress-anim {
        from { width: 100%; }
        to { width: 0%; }
      }

      .haiven-toast.haiven-toast-success .haiven-toast-progress-bar {
        background: #22c55e;
      }

      .haiven-toast.haiven-toast-error .haiven-toast-progress-bar {
        background: #ef4444;
      }
    `;

    document.head.appendChild(style);
  }

  showToast(options) {
    const {
      type = 'info', // success, error, warning, info
      title = '',
      message = '',
      duration = this.settings.toastDuration,
      dismissible = true
    } = options;

    // Remove oldest toast if we've hit the limit
    if (this.toastQueue.length >= this.settings.maxToasts) {
      const oldestToast = this.toastQueue.shift();
      if (oldestToast && oldestToast.element) {
        this.dismissToast(oldestToast.element);
      }
    }

    // Create toast element
    const toast = document.createElement('div');
    toast.className = `haiven-toast haiven-toast-${type}`;

    const icons = {
      success: '✓',
      error: '✕',
      warning: '⚠',
      info: 'ℹ'
    };

    toast.innerHTML = `
      <div class="haiven-toast-icon">${icons[type]}</div>
      <div class="haiven-toast-content">
        ${title ? `<div class="haiven-toast-title">${this.escapeHtml(title)}</div>` : ''}
        <div class="haiven-toast-message">${this.escapeHtml(message)}</div>
      </div>
      ${dismissible ? '<button class="haiven-toast-close">×</button>' : ''}
      ${duration > 0 ? `<div class="haiven-toast-progress"><div class="haiven-toast-progress-bar" style="animation-duration: ${duration}ms"></div></div>` : ''}
    `;

    // Add to container
    this.toastContainer.appendChild(toast);

    // Add to queue
    const toastData = { element: toast, timeoutId: null };
    this.toastQueue.push(toastData);

    // Attach close button listener
    if (dismissible) {
      const closeBtn = toast.querySelector('.haiven-toast-close');
      closeBtn.addEventListener('click', () => this.dismissToast(toast));
    }

    // Auto-dismiss after duration
    if (duration > 0) {
      toastData.timeoutId = setTimeout(() => {
        this.dismissToast(toast);
      }, duration);
    }

    return toast;
  }

  dismissToast(toastElement) {
    // Add exit animation
    toastElement.classList.add('haiven-toast-out');

    // Remove from queue
    this.toastQueue = this.toastQueue.filter((t) => t.element !== toastElement);

    // Remove from DOM after animation
    setTimeout(() => {
      if (toastElement.parentNode) {
        toastElement.remove();
      }
    }, 300);
  }

  // Convenience methods for different toast types
  showSuccessToast(title, message) {
    return this.showToast({ type: 'success', title, message });
  }

  showErrorToast(title, message) {
    return this.showToast({ type: 'error', title, message, duration: 5000 });
  }

  showWarningToast(title, message) {
    return this.showToast({ type: 'warning', title, message });
  }

  showInfoToast(title, message) {
    return this.showToast({ type: 'info', title, message });
  }

  attachInputListeners() {
    // Listen for input in text areas and content editables
    document.addEventListener(
      'input',
      (e) => {
        const { target } = e;

        // Check if it's a text input area
        if (
          target.tagName === 'TEXTAREA' ||
          target.isContentEditable ||
          target.tagName === 'INPUT'
        ) {
          this.handleInput(target);
        }
      },
      true
    );
  }

  handleInput(inputElement) {
    // Get current text
    const text = inputElement.value || inputElement.textContent || '';

    // Clear previous timeout
    if (this.searchTimeout) {
      clearTimeout(this.searchTimeout);
    }

    // Only search if text is long enough
    if (text.length < this.settings.minChars) {
      this.hide();
      return;
    }

    // Debounce - wait for user to stop typing
    this.searchTimeout = setTimeout(() => {
      this.searchMemories(text, inputElement);
    }, this.settings.debounceMs);
  }

  async searchMemories(query, inputElement) {
    // Skip if query is the same as last time
    if (query === this.lastQuery) return;
    this.lastQuery = query;

    // Check if we're in cooldown period after injection
    const timeSinceInjection = Date.now() - this.lastInjectionTime;
    if (this.lastInjectionTime > 0 && timeSinceInjection < this.settings.sessionCooldownMs) {
      const remainingSecs = Math.round(
        (this.settings.sessionCooldownMs - timeSinceInjection) / 1000
      );
      console.log(`[Haiven Predictive] In cooldown period - ${remainingSecs}s remaining`);
      return;
    }

    // Check if we've already injected memories for similar context
    // Don't show overlay again unless query has changed significantly
    if (this.lastInjectedQuery) {
      const similarity = this.calculateQuerySimilarity(query, this.lastInjectedQuery);
      if (similarity > 1 - this.settings.minQueryChange) {
        console.log('[Haiven Predictive] Skipping overlay - already injected for similar context');
        return;
      }
    }

    // NO loading state - search silently, only show if we have results
    try {
      // Get current userId from storage
      const { userId } = await window.browserAPI.storage.local.get(['userId']);
      if (!userId) {
        console.log('[Haiven Predictive] No userId found - user not logged in');
        return;
      }

      console.log('[Haiven Predictive] Searching for:', query.substring(0, 50) + '...');

      // Get session context (visible conversation) for better relevance
      const sessionContext = this.getSessionContext();

      // Get user's top topics for personalization
      const topTopics = this.getTopTopics(5);

      // Combine user input with session context for richer search
      let enrichedContext = query;
      if (sessionContext) {
        enrichedContext += `\n\n[Conversation Context]:\n${sessionContext}`;
      }
      if (topTopics.length > 0) {
        enrichedContext += `\n\n[User Interests]: ${topTopics.join(', ')}`;
      }

      let memories = [];

      // Try smart-context endpoint first
      try {
        const response = await fetch('https://api.safehaiven.com/api/memory/smart-context', {
          method: 'POST',
          headers: await window.getAuthHeaders(),
          body: JSON.stringify({
            userId,
            currentContext: enrichedContext,
            limit: this.settings.maxResults,
            platform: this.detectPlatform(),
            userBehavior: {
              usedCount: this.userBehavior.usedMemoryIds.length,
              dismissedCount: this.userBehavior.dismissedMemoryIds.length,
              topTopics,
              platformStats: this.userBehavior.platformStats
            }
          })
        });

        if (response.ok) {
          const data = await response.json();
          memories = data.memories || [];
          console.log('[Haiven Predictive] smart-context returned', memories.length, 'memories');
        } else {
          console.warn(
            '[Haiven Predictive] smart-context failed:',
            response.status,
            '- trying fallback'
          );
          throw new Error('smart-context failed');
        }
      } catch (smartContextError) {
        // Fallback to recall endpoint
        console.log('[Haiven Predictive] Using fallback: /api/chatgpt/memory/recall');
        try {
          const fallbackResponse = await fetch(
            'https://api.safehaiven.com/api/chatgpt/memory/recall',
            {
              method: 'POST',
              headers: await window.getAuthHeaders(),
              body: JSON.stringify({
                query: query.substring(0, 500),
                limit: this.settings.maxResults
              })
            }
          );

          if (fallbackResponse.ok) {
            const fallbackData = await fallbackResponse.json();
            memories = fallbackData.memories || [];
            console.log(
              '[Haiven Predictive] recall fallback returned',
              memories.length,
              'memories'
            );
          } else {
            console.error('[Haiven Predictive] Fallback also failed:', fallbackResponse.status);
          }
        } catch (fallbackError) {
          console.error('[Haiven Predictive] Fallback error:', fallbackError.message);
        }
      }

      // Filter out low-relevance results
      const filteredMemories = memories.filter((m) => {
        const relevance = m.relevance || m.relevanceScore || m.similarity || 0.5;
        return relevance >= this.settings.minRelevance;
      });

      console.log(
        `[Haiven Predictive] Found ${memories.length} memories, ${filteredMemories.length} above ${this.settings.minRelevance * 100}% relevance threshold`
      );

      // ONLY show overlay if we have relevant memories - otherwise stay hidden
      if (filteredMemories.length > 0) {
        this.currentMemories = filteredMemories;
        this.currentInputElement = inputElement;
        this.show(filteredMemories);
      } else {
        // No results? Stay quiet - don't interrupt the user
        console.log('[Haiven Predictive] No memories above threshold - staying hidden');
        this.hide();
      }
    } catch (error) {
      console.error('[Haiven Predictive] Search error:', error);
      // Silent failure - don't show anything
    }
  }

  // Calculate similarity between two queries (0-1, 1 = identical)
  calculateQuerySimilarity(query1, query2) {
    if (!query1 || !query2) return 0;

    // Normalize queries
    const normalize = (str) =>
      str
        .toLowerCase()
        .replace(/[^\w\s]/g, '')
        .trim();
    const q1 = normalize(query1);
    const q2 = normalize(query2);

    // Simple word-based similarity
    const words1 = new Set(q1.split(/\s+/));
    const words2 = new Set(q2.split(/\s+/));

    const intersection = new Set([...words1].filter((x) => words2.has(x)));
    const union = new Set([...words1, ...words2]);

    return union.size > 0 ? intersection.size / union.size : 0;
  }

  // Show loading state
  showLoading() {
    const contentEl = this.overlay.querySelector('#haiven-memories');
    contentEl.innerHTML = `
      <div class="haiven-loading">
        <div class="haiven-spinner"></div>
        <div class="haiven-loading-text">Searching memories...</div>
      </div>
    `;
    this.isLoading = true;
    this.overlay.classList.remove('hidden');
    this.isVisible = true;
  }

  // Hide loading state
  hideLoading() {
    this.isLoading = false;
  }

  // Show empty state with helpful messaging
  showEmptyState(context = 'search') {
    this.hideLoading();
    const contentEl = this.overlay.querySelector('#haiven-memories');
    const countEl = this.overlay.querySelector('#haiven-count');

    // Update count
    countEl.textContent = '0';

    // Determine empty state messaging based on context
    let icon, title, message, tips;

    if (context === 'search') {
      icon = '🔍';
      title = 'No Memories Found';
      message =
        "We couldn't find any relevant memories for this context. Keep typing and we'll search again.";
      tips = [
        { icon: '💡', text: 'Add more specific details to help us find related memories' },
        { icon: '⌨️', text: 'Try typing at least 20-30 characters for better results' },
        { icon: '🎯', text: "Mention topics, projects, or people you've discussed before" }
      ];
    } else if (context === 'first-time') {
      icon = '✨';
      title = 'Start Building Your Memory';
      message =
        "You haven't saved any memories yet. Start conversations across ChatGPT, Claude, Grok, and we'll remember the important parts.";
      tips = [
        { icon: '💬', text: 'Have natural conversations - we auto-capture key insights' },
        { icon: '🌍', text: 'Memories are organized into 8 life Worlds automatically' },
        { icon: '🔗', text: 'Context follows you across all AI platforms seamlessly' }
      ];
    }

    contentEl.innerHTML = `
      <div class="haiven-empty-state">
        <div class="haiven-empty-icon">${icon}</div>
        <div class="haiven-empty-title">${title}</div>
        <div class="haiven-empty-message">${message}</div>
        <div class="haiven-empty-tips">
          <div class="haiven-empty-tips-title">💡 Tips</div>
          ${tips
            .map(
              (tip) => `
            <div class="haiven-empty-tip">
              <div class="haiven-empty-tip-icon">${tip.icon}</div>
              <div>${tip.text}</div>
            </div>
          `
            )
            .join('')}
        </div>
      </div>
    `;

    // Show overlay
    this.overlay.classList.remove('hidden');
    this.isVisible = true;

    // Auto-hide after 5 seconds for search context
    if (context === 'search') {
      setTimeout(() => {
        if (this.isVisible && contentEl.querySelector('.haiven-empty-state')) {
          this.hide();
        }
      }, 5000);
    }
  }

  // Update keyboard selection highlighting
  updateSelection() {
    const cards = this.overlay.querySelectorAll('.haiven-memory-card');
    cards.forEach((card, index) => {
      if (index === this.selectedIndex) {
        card.classList.add('haiven-selected');
        card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
      } else {
        card.classList.remove('haiven-selected');
      }
    });
  }

  // Show memory preview on hover
  showPreview(memory, cardElement) {
    // Remove existing preview
    this.hidePreview();

    const preview = document.createElement('div');
    preview.className = 'haiven-memory-preview';
    preview.innerHTML = `
      <div class="haiven-preview-header">
        <span class="haiven-preview-platform">${memory.metadata?.platform || memory.platform || 'Memory'}</span>
        <span class="haiven-preview-time">${this.formatTimestamp(memory.created_at)}</span>
      </div>
      <div class="haiven-preview-content">${this.escapeHtml(memory.content)}</div>
      ${memory.world ? `<div class="haiven-preview-world">World: ${memory.world}</div>` : ''}
    `;

    const rect = cardElement.getBoundingClientRect();
    const previewWidth = 400; // Max width from CSS
    const spaceOnRight = window.innerWidth - rect.right;
    const spaceOnLeft = rect.left;

    // Position on left if not enough space on right
    let leftPos;
    if (spaceOnRight >= previewWidth + 20) {
      leftPos = rect.right + 16;
    } else if (spaceOnLeft >= previewWidth + 20) {
      leftPos = rect.left - previewWidth - 16;
    } else {
      // Not enough space on either side, don't show preview
      return;
    }

    // Also check vertical positioning
    const previewHeight = 200; // Approximate
    let topPos = rect.top;
    if (topPos + previewHeight > window.innerHeight) {
      topPos = Math.max(10, window.innerHeight - previewHeight - 10);
    }

    preview.style.top = `${topPos}px`;
    preview.style.left = `${leftPos}px`;

    document.body.appendChild(preview);
    this.currentPreview = preview;
  }

  // Hide memory preview
  hidePreview() {
    if (this.currentPreview) {
      this.currentPreview.remove();
      this.currentPreview = null;
    }
  }

  show(memories) {
    this.hideLoading();
    const countEl = this.overlay.querySelector('#haiven-count');
    const contentEl = this.overlay.querySelector('#haiven-memories');

    // Update count
    countEl.textContent = memories.length;

    const _currentPlatform = this.detectPlatform(); // Reserved for platform-specific rendering

    // Render memory cards with clean, natural layout
    contentEl.innerHTML = memories
      .map((memory, index) => {
        const confidence = memory.relevance ? Math.round(memory.relevance * 100) : 85;
        const isRecentlyInjected = this.recentlyInjectedIds.has(memory.id);
        const memoryPlatform = memory.metadata?.platform || memory.platform || 'Unknown';
        const recencyClass = this.getRecencyClass(memory.created_at);

        return `
        <div class="haiven-memory-card ${recencyClass}${index === 0 ? ' haiven-selected' : ''}" data-memory-index="${index}">
          ${isRecentlyInjected ? '<div class="haiven-recently-injected">Added</div>' : ''}
          <div class="haiven-memory-header">
            <div class="haiven-memory-source">
              <span class="haiven-platform-icon">${this.getPlatformIcon(memoryPlatform)}</span>
              <span class="haiven-meta">
                <span class="haiven-platform">${memoryPlatform}</span> · ${this.formatTimestamp(memory.created_at)}
              </span>
            </div>
            <div class="haiven-confidence">
              <span class="haiven-confidence-text">${confidence}%</span>
            </div>
          </div>
          <div class="haiven-memory-content">${this.escapeHtml(memory.content)}</div>
          <div class="haiven-memory-footer">
            ${memory.world ? `<span class="haiven-memory-world">${memory.world}</span>` : '<span></span>'}
            <button class="haiven-add-btn" data-memory-index="${index}">
              Use <span class="haiven-btn-shortcut">${index + 1}</span>
            </button>
          </div>
        </div>
      `;
      })
      .join('');

    // Attach click listeners to Add buttons
    contentEl.querySelectorAll('.haiven-add-btn').forEach((btn) => {
      btn.addEventListener('click', (e) => {
        const index = parseInt(e.target.closest('.haiven-add-btn').dataset.memoryIndex);
        this.addMemory(index);
      });
    });

    // Add hover preview listeners
    contentEl.querySelectorAll('.haiven-memory-card').forEach((card, index) => {
      card.addEventListener('mouseenter', () => {
        this.showPreview(memories[index], card);
      });
      card.addEventListener('mouseleave', () => {
        this.hidePreview();
      });
    });

    // Reset selection
    this.selectedIndex = 0;

    // Show overlay
    this.overlay.classList.remove('hidden');
    this.isVisible = true;

    // Ensure overlay stays within viewport bounds
    this.positionOverlayInViewport();
  }

  hide(userDismissed = false) {
    // Track dismissal if user explicitly closed without using any memory
    if (userDismissed && this.currentMemories.length > 0) {
      this.trackDismissed(this.currentMemories);
    }

    this.overlay.classList.add('hidden');
    this.isVisible = false;
    this.lastQuery = '';
  }

  async addMemory(index) {
    const memory = this.currentMemories[index];
    if (!memory || !this.currentInputElement) return;

    // Track that user actually used this memory (positive signal)
    this.trackMemoryUsed(memory);

    // Mark as recently injected
    this.recentlyInjectedIds.add(memory.id);
    setTimeout(() => {
      this.recentlyInjectedIds.delete(memory.id);
    }, this.settings.recentInjectionTimeout);

    try {
      // Get userId from storage
      const { userId } = await window.browserAPI.storage.local.get(['userId']);
      if (!userId) return;

      // Call smart-context API to get Groq-summarized version
      const response = await fetch('https://api.safehaiven.com/api/memory/smart-context', {
        method: 'POST',
        headers: await window.getAuthHeaders(),
        body: JSON.stringify({
          userId,
          query: memory.content.substring(0, 200), // Use first 200 chars as query context
          memoryIds: [memory.id],
          platform: this.detectPlatform(),
          maxTokens: 500
        })
      });

      if (!response.ok) {
        console.error('[Haiven Predictive] Smart context failed:', response.status);
        // Fallback to formatted content
        const context = this.formatMemoryForInjection(memory);
        this.insertText(this.currentInputElement, context);
      } else {
        const data = await response.json();
        const summarizedContext = data.context || memory.content;

        // Insert Groq-summarized context with formatting
        const formattedContext = `📝 **Context from Haiven Memory** (${this.formatTimestamp(memory.created_at)}):\n\n${summarizedContext}\n\n---\n\n`;
        this.insertText(this.currentInputElement, formattedContext);
      }

      // Show success toast - prominent confirmation
      this.showSuccessToast('Added to prompt', 'Memory context is ready to use');

      // Mark this query as injected - don't show overlay again for similar context
      this.lastInjectedQuery = this.lastQuery;
      this.lastInjectionTime = Date.now(); // Start cooldown timer

      // Hide overlay
      setTimeout(() => this.hide(), 300);
    } catch (error) {
      console.error('[Haiven Predictive] Add memory error:', error);
      // Fallback to formatted content
      const context = this.formatMemoryForInjection(memory);
      this.insertText(this.currentInputElement, context);

      // Show success toast even for fallback
      this.showSuccessToast('Added to prompt', 'Memory context is ready to use');

      // Mark this query as injected - don't show overlay again for similar context
      this.lastInjectedQuery = this.lastQuery;
      this.lastInjectionTime = Date.now(); // Start cooldown timer

      setTimeout(() => this.hide(), 300);
    }
  }

  async addAllMemories() {
    if (this.currentMemories.length === 0 || !this.currentInputElement) return;

    // Mark all as recently injected
    this.currentMemories.forEach((memory) => {
      this.recentlyInjectedIds.add(memory.id);
      setTimeout(() => {
        this.recentlyInjectedIds.delete(memory.id);
      }, this.settings.recentInjectionTimeout);
    });

    try {
      // Get userId from storage
      const { userId } = await window.browserAPI.storage.local.get(['userId']);
      if (!userId) return;

      // Combine all memory content as query context
      const combinedQuery = this.currentMemories.map((m) => m.content.substring(0, 100)).join(' ');
      const memoryIds = this.currentMemories.map((m) => m.id);

      // Call smart-context API to get Groq-summarized version of all memories
      const response = await fetch('https://api.safehaiven.com/api/memory/smart-context', {
        method: 'POST',
        headers: await window.getAuthHeaders(),
        body: JSON.stringify({
          userId,
          query: combinedQuery.substring(0, 300),
          memoryIds,
          platform: this.detectPlatform(),
          maxTokens: 1000
        })
      });

      if (!response.ok) {
        console.error('[Haiven Predictive] Smart context failed:', response.status);
        // Fallback to formatted content
        const allContext = this.formatMultipleMemoriesForInjection(this.currentMemories);
        this.insertText(this.currentInputElement, allContext);
      } else {
        const data = await response.json();
        const summarizedContext =
          data.context || this.currentMemories.map((m) => m.content).join('\n\n');

        // Insert Groq-summarized context with clean formatting
        const count = this.currentMemories.length;
        const formattedContext = `📚 **Context from ${count} Haiven ${count > 1 ? 'Memories' : 'Memory'}:**\n\n${summarizedContext}\n\n---\n\n`;
        this.insertText(this.currentInputElement, formattedContext);
      }

      // Show success toast with count
      this.showSuccessToast(
        `${this.currentMemories.length} memories added`,
        'All context is ready to use'
      );

      // Mark this query as injected - don't show overlay again for similar context
      this.lastInjectedQuery = this.lastQuery;
      this.lastInjectionTime = Date.now(); // Start cooldown timer

      // Hide overlay
      setTimeout(() => this.hide(), 300);
    } catch (error) {
      console.error('[Haiven Predictive] Add all memories error:', error);
      // Fallback to formatted content
      const allContext = this.formatMultipleMemoriesForInjection(this.currentMemories);
      this.insertText(this.currentInputElement, allContext);

      // Show success toast even for fallback
      this.showSuccessToast(
        `${this.currentMemories.length} memories added`,
        'All context is ready to use'
      );

      // Mark this query as injected - don't show overlay again for similar context
      this.lastInjectedQuery = this.lastQuery;
      this.lastInjectionTime = Date.now(); // Start cooldown timer

      setTimeout(() => this.hide(), 300);
    }
  }

  insertText(element, text) {
    if (element.tagName === 'TEXTAREA' || element.tagName === 'INPUT') {
      // For textarea/input
      const start = element.selectionStart;
      const end = element.selectionEnd;
      const currentValue = element.value;

      element.value = currentValue.substring(0, start) + text + currentValue.substring(end);
      element.selectionStart = element.selectionEnd = start + text.length;

      // Trigger input event
      element.dispatchEvent(new Event('input', { bubbles: true }));
    } else if (element.isContentEditable) {
      // For contentEditable
      const selection = window.getSelection();
      const range = selection.getRangeAt(0);
      range.deleteContents();

      const textNode = document.createTextNode(text);
      range.insertNode(textNode);
      range.setStartAfter(textNode);
      range.setEndAfter(textNode);
      selection.removeAllRanges();
      selection.addRange(range);

      // Trigger input event
      element.dispatchEvent(new Event('input', { bubbles: true }));
    }

    // Focus the element
    element.focus();
  }

  attachKeyboardShortcuts() {
    document.addEventListener('keydown', (e) => {
      // Ctrl+Shift+H to toggle mini dashboard (global shortcut)
      if ((e.ctrlKey || e.metaKey) && e.shiftKey && e.key.toLowerCase() === 'h') {
        this.toggleMiniDashboard();
        e.preventDefault();
        return;
      }

      // Esc to close mini dashboard or overlay
      if (e.key === 'Escape') {
        const miniDash = document.getElementById('haiven-mini-dashboard');
        if (miniDash) {
          miniDash.remove();
          e.preventDefault();
          return;
        }
        if (this.isVisible) {
          this.hide(true);
          e.preventDefault();
          return;
        }
      }

      // Rest of shortcuts only apply when overlay is visible
      if (!this.isVisible) return;

      // Cmd/Ctrl + Enter to add all
      if ((e.metaKey || e.ctrlKey) && e.key === 'Enter') {
        this.addAllMemories();
        e.preventDefault();
        return;
      }

      // Number keys 1-3 to select specific memory
      if (e.key >= '1' && e.key <= '3') {
        const index = parseInt(e.key) - 1;
        if (index < this.currentMemories.length) {
          this.addMemory(index);
          e.preventDefault();
        }
        return;
      }

      // Arrow keys for navigation
      if (e.key === 'ArrowDown') {
        this.selectedIndex = Math.min(this.selectedIndex + 1, this.currentMemories.length - 1);
        this.updateSelection();
        e.preventDefault();
      } else if (e.key === 'ArrowUp') {
        this.selectedIndex = Math.max(this.selectedIndex - 1, 0);
        this.updateSelection();
        e.preventDefault();
      } else if (e.key === 'Enter' && this.selectedIndex >= 0) {
        this.addMemory(this.selectedIndex);
        e.preventDefault();
      }
    });
  }

  makeDraggable() {
    const header = this.overlay.querySelector('.haiven-overlay-header');
    let isDragging = false;
    let currentX, currentY, initialX, initialY;

    header.addEventListener('mousedown', (e) => {
      isDragging = true;
      initialX = e.clientX - this.overlay.offsetLeft;
      initialY = e.clientY - this.overlay.offsetTop;
    });

    document.addEventListener('mousemove', (e) => {
      if (!isDragging) return;

      e.preventDefault();
      currentX = e.clientX - initialX;
      currentY = e.clientY - initialY;

      this.overlay.style.left = `${currentX}px`;
      this.overlay.style.top = `${currentY}px`;
      this.overlay.style.right = 'auto';
      this.overlay.style.bottom = 'auto';
    });

    document.addEventListener('mouseup', () => {
      isDragging = false;
    });
  }

  positionOverlayInViewport() {
    // Ensure overlay stays within viewport boundaries
    // This prevents the overlay from appearing off-screen on small viewports

    // Get viewport dimensions
    const viewportWidth = window.innerWidth;
    const viewportHeight = window.innerHeight;

    // Get overlay dimensions after it's been rendered
    const overlayRect = this.overlay.getBoundingClientRect();
    const overlayWidth = overlayRect.width;
    const overlayHeight = overlayRect.height;

    // Calculate default position (bottom-right with 24px margin)
    const defaultRight = 24;
    const defaultBottom = 24;

    // Check if overlay fits in default position
    const fitsHorizontally = viewportWidth - defaultRight >= overlayWidth;
    const fitsVertically = viewportHeight - defaultBottom >= overlayHeight;

    // If overlay has been dragged (has explicit left/top), don't reposition
    if (this.overlay.style.left && this.overlay.style.left !== 'auto') {
      return;
    }

    // Reset to default positioning if it fits
    if (fitsHorizontally && fitsVertically) {
      this.overlay.style.right = `${defaultRight}px`;
      this.overlay.style.bottom = `${defaultBottom}px`;
      this.overlay.style.left = 'auto';
      this.overlay.style.top = 'auto';
      return;
    }

    // Calculate adjusted position to keep overlay on screen
    let adjustedRight = defaultRight;
    let adjustedBottom = defaultBottom;

    // Adjust horizontal position if needed
    if (!fitsHorizontally) {
      adjustedRight = Math.max(8, viewportWidth - overlayWidth - 8);
    }

    // Adjust vertical position if needed
    if (!fitsVertically) {
      adjustedBottom = Math.max(8, viewportHeight - overlayHeight - 8);
    }

    // Apply adjusted positioning
    this.overlay.style.right = `${adjustedRight}px`;
    this.overlay.style.bottom = `${adjustedBottom}px`;
    this.overlay.style.left = 'auto';
    this.overlay.style.top = 'auto';
  }

  // ========================================
  // MINI-DASHBOARD - Quick context at a glance
  // ========================================

  createDashboardButton() {
    // Create floating Haiven button
    this.dashboardButton = document.createElement('div');
    this.dashboardButton.id = 'haiven-dashboard-btn';
    this.dashboardButton.className = 'haiven-dashboard-btn';
    this.dashboardButton.innerHTML = `
      <span class="haiven-btn-icon">🧠</span>
      <span class="haiven-btn-badge" id="haiven-task-badge" style="display: none;">0</span>
    `;
    this.dashboardButton.title = 'Haiven Memory (Ctrl+Shift+H)';

    // Add styles for dashboard button
    this.injectDashboardStyles();

    // Add to page
    document.body.appendChild(this.dashboardButton);

    // Attach click listener
    this.dashboardButton.addEventListener('click', () => this.toggleMiniDashboard());
  }

  injectDashboardStyles() {
    if (document.getElementById('haiven-dashboard-btn-styles')) return;

    const style = document.createElement('style');
    style.id = 'haiven-dashboard-btn-styles';
    style.textContent = `
      /* Floating Haiven Button */
      .haiven-dashboard-btn {
        position: fixed;
        bottom: 24px;
        left: 24px;
        width: 48px;
        height: 48px;
        background: rgba(17, 17, 23, 0.95);
        backdrop-filter: blur(20px);
        border: 1px solid rgba(255, 255, 255, 0.1);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        cursor: pointer;
        z-index: 999998;
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        box-shadow: 0 4px 16px rgba(0, 0, 0, 0.3);
      }

      .haiven-dashboard-btn:hover {
        transform: scale(1.1);
        border-color: rgba(99, 102, 241, 0.5);
        box-shadow: 0 8px 24px rgba(0, 0, 0, 0.4);
      }

      .haiven-btn-icon {
        font-size: 20px;
      }

      .haiven-btn-badge {
        position: absolute;
        top: -4px;
        right: -4px;
        min-width: 18px;
        height: 18px;
        background: #ef4444;
        color: white;
        font-size: 10px;
        font-weight: 600;
        border-radius: 9px;
        display: flex;
        align-items: center;
        justify-content: center;
        padding: 0 4px;
        font-family: -apple-system, BlinkMacSystemFont, sans-serif;
      }

      /* Mini Dashboard Panel */
      .haiven-mini-dashboard {
        position: fixed;
        bottom: 80px;
        left: 24px;
        width: 320px;
        background: rgba(17, 17, 23, 0.98);
        backdrop-filter: blur(40px);
        border: 1px solid rgba(255, 255, 255, 0.08);
        border-radius: 16px;
        overflow: hidden;
        z-index: 999999;
        font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', sans-serif;
        color: #ffffff;
        animation: haiven-dash-in 0.3s ease;
        box-shadow: 0 24px 48px -12px rgba(0, 0, 0, 0.5);
      }

      .haiven-mini-dashboard.hidden {
        display: none;
      }

      @keyframes haiven-dash-in {
        from {
          opacity: 0;
          transform: translateY(20px) scale(0.95);
        }
        to {
          opacity: 1;
          transform: translateY(0) scale(1);
        }
      }

      .haiven-dash-header {
        padding: 16px;
        border-bottom: 1px solid rgba(255, 255, 255, 0.06);
        display: flex;
        align-items: center;
        justify-content: space-between;
      }

      .haiven-dash-title {
        font-size: 14px;
        font-weight: 600;
        color: rgba(255, 255, 255, 0.9);
      }

      .haiven-dash-close {
        width: 24px;
        height: 24px;
        border: none;
        background: transparent;
        color: rgba(255, 255, 255, 0.4);
        cursor: pointer;
        border-radius: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 14px;
        transition: all 0.2s;
      }

      .haiven-dash-close:hover {
        background: rgba(255, 255, 255, 0.1);
        color: rgba(255, 255, 255, 0.8);
      }

      .haiven-dash-stats {
        display: grid;
        grid-template-columns: repeat(3, 1fr);
        gap: 12px;
        padding: 16px;
        border-bottom: 1px solid rgba(255, 255, 255, 0.06);
      }

      .haiven-stat {
        text-align: center;
      }

      .haiven-stat-value {
        font-size: 20px;
        font-weight: 700;
        color: #ffffff;
      }

      .haiven-stat-label {
        font-size: 10px;
        color: rgba(255, 255, 255, 0.5);
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-top: 2px;
      }

      .haiven-dash-section {
        padding: 12px 16px;
        border-bottom: 1px solid rgba(255, 255, 255, 0.04);
      }

      .haiven-dash-section:last-child {
        border-bottom: none;
      }

      .haiven-section-title {
        font-size: 11px;
        font-weight: 600;
        color: rgba(255, 255, 255, 0.5);
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 8px;
        display: flex;
        align-items: center;
        gap: 6px;
      }

      .haiven-section-item {
        font-size: 13px;
        color: rgba(255, 255, 255, 0.8);
        padding: 6px 0;
        border-bottom: 1px solid rgba(255, 255, 255, 0.03);
        display: flex;
        align-items: flex-start;
        gap: 8px;
      }

      .haiven-section-item:last-child {
        border-bottom: none;
      }

      .haiven-item-icon {
        flex-shrink: 0;
        width: 18px;
        text-align: center;
      }

      .haiven-item-text {
        flex: 1;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
      }

      .haiven-dash-empty {
        text-align: center;
        padding: 24px 16px;
        color: rgba(255, 255, 255, 0.4);
        font-size: 13px;
      }

      .haiven-dash-footer {
        padding: 12px 16px;
        background: rgba(0, 0, 0, 0.2);
        display: flex;
        justify-content: center;
      }

      .haiven-dash-link {
        font-size: 12px;
        color: #6366f1;
        text-decoration: none;
        font-weight: 500;
        transition: color 0.2s;
      }

      .haiven-dash-link:hover {
        color: #818cf8;
      }
    `;

    document.head.appendChild(style);
  }

  async prefetchDashboardData() {
    try {
      const { userId } = await window.browserAPI.storage.local.get(['userId']);
      if (!userId) return;

      console.log('[Haiven] Prefetching dashboard data...');

      // Fetch stats
      const statsRes = await fetch(`https://api.safehaiven.com/api/stats?userId=${userId}`, {
        headers: await window.getAuthHeaders()
      });
      const stats = statsRes.ok ? await statsRes.json() : { total_memories: 0 };

      // Fetch open loops
      let openLoops = [];
      try {
        const loopsRes = await fetch(
          `https://api.safehaiven.com/api/open-loops?userId=${userId}&status=open&limit=3`,
          { headers: await window.getAuthHeaders() }
        );
        if (loopsRes.ok) {
          const loopsData = await loopsRes.json();
          openLoops = loopsData.open_loops || [];
        }
      } catch (e) {
        console.log('[Haiven] Could not fetch open loops');
      }

      // Fetch active threads
      let activeThreads = [];
      try {
        const threadsRes = await fetch(
          `https://api.safehaiven.com/api/threads/user/${userId}?isActive=true&limit=3`,
          { headers: await window.getAuthHeaders() }
        );
        if (threadsRes.ok) {
          activeThreads = await threadsRes.json();
        }
      } catch (e) {
        console.log('[Haiven] Could not fetch threads');
      }

      // Fetch recent memories
      let recentMemories = [];
      try {
        const memoriesRes = await fetch(
          `https://api.safehaiven.com/api/memory/recent?userId=${userId}&limit=3`,
          { headers: await window.getAuthHeaders() }
        );
        if (memoriesRes.ok) {
          const memoriesData = await memoriesRes.json();
          recentMemories = memoriesData.memories || memoriesData || [];
        }
      } catch (e) {
        console.log('[Haiven] Could not fetch memories');
      }

      this.dashboardData = {
        stats,
        openLoops,
        activeThreads,
        recentMemories,
        fetchedAt: Date.now()
      };

      // Update badge
      this.updateTaskBadge(openLoops.length);

      console.log('[Haiven] Dashboard data prefetched:', {
        memories: stats.total_memories || 0,
        openLoops: openLoops.length,
        threads: activeThreads.length
      });
    } catch (e) {
      console.error('[Haiven] Failed to prefetch dashboard data:', e);
    }
  }

  updateTaskBadge(count) {
    const badge = document.getElementById('haiven-task-badge');
    if (badge) {
      if (count > 0) {
        badge.textContent = count > 9 ? '9+' : count;
        badge.style.display = 'flex';
      } else {
        badge.style.display = 'none';
      }
    }
  }

  toggleMiniDashboard() {
    const existing = document.getElementById('haiven-mini-dashboard');
    if (existing) {
      existing.remove();
      return;
    }

    this.showMiniDashboard();
  }

  showMiniDashboard() {
    // Remove existing
    const existing = document.getElementById('haiven-mini-dashboard');
    if (existing) existing.remove();

    // Refresh data if stale (> 2 minutes old)
    if (!this.dashboardData || Date.now() - this.dashboardData.fetchedAt > 120000) {
      this.prefetchDashboardData();
    }

    const data = this.dashboardData || {
      stats: { total_memories: 0 },
      openLoops: [],
      activeThreads: [],
      recentMemories: []
    };

    const platform = this.detectPlatform();

    const dashboard = document.createElement('div');
    dashboard.id = 'haiven-mini-dashboard';
    dashboard.className = 'haiven-mini-dashboard';

    dashboard.innerHTML = `
      <div class="haiven-dash-header">
        <span class="haiven-dash-title">🧠 Haiven Memory</span>
        <button class="haiven-dash-close" id="haiven-dash-close">×</button>
      </div>

      <div class="haiven-dash-stats">
        <div class="haiven-stat">
          <div class="haiven-stat-value">${data.stats.total_memories || 0}</div>
          <div class="haiven-stat-label">Memories</div>
        </div>
        <div class="haiven-stat">
          <div class="haiven-stat-value">${data.openLoops.length}</div>
          <div class="haiven-stat-label">Open Tasks</div>
        </div>
        <div class="haiven-stat">
          <div class="haiven-stat-value">${data.activeThreads.length}</div>
          <div class="haiven-stat-label">Active</div>
        </div>
      </div>

      ${
        data.activeThreads.length > 0
          ? `
        <div class="haiven-dash-section">
          <div class="haiven-section-title">💭 Where you left off</div>
          ${data.activeThreads
            .slice(0, 3)
            .map(
              (t) => `
            <div class="haiven-section-item">
              <span class="haiven-item-icon">📌</span>
              <span class="haiven-item-text">${this.escapeHtml(t.title || 'Untitled')}</span>
            </div>
          `
            )
            .join('')}
        </div>
      `
          : ''
      }

      ${
        data.openLoops.length > 0
          ? `
        <div class="haiven-dash-section">
          <div class="haiven-section-title">🔴 Needs Attention</div>
          ${data.openLoops
            .slice(0, 3)
            .map((loop) => {
              const icon =
                loop.priority === 'urgent' ? '🚨' : loop.priority === 'high' ? '⚡' : '📋';
              return `
              <div class="haiven-section-item">
                <span class="haiven-item-icon">${icon}</span>
                <span class="haiven-item-text">${this.escapeHtml(loop.description)}</span>
              </div>
            `;
            })
            .join('')}
        </div>
      `
          : ''
      }

      ${
        data.recentMemories.length > 0
          ? `
        <div class="haiven-dash-section">
          <div class="haiven-section-title">📝 Recent Memories</div>
          ${data.recentMemories
            .slice(0, 2)
            .map(
              (m) => `
            <div class="haiven-section-item">
              <span class="haiven-item-icon">💡</span>
              <span class="haiven-item-text">${this.escapeHtml(m.content.substring(0, 50))}${m.content.length > 50 ? '...' : ''}</span>
            </div>
          `
            )
            .join('')}
        </div>
      `
          : ''
      }

      ${
        data.stats.total_memories === 0
          ? `
        <div class="haiven-dash-empty">
          <p>No memories yet!</p>
          <p style="font-size: 11px; margin-top: 8px;">Type in ${platform} and your important context will be remembered.</p>
        </div>
      `
          : ''
      }

      <div class="haiven-dash-footer">
        <a href="https://www.safehaiven.com/dashboard" target="_blank" class="haiven-dash-link">
          Open Full Dashboard →
        </a>
      </div>
    `;

    document.body.appendChild(dashboard);

    // Attach close listener
    document.getElementById('haiven-dash-close').addEventListener('click', () => {
      dashboard.remove();
    });

    // Close on click outside
    const closeOnOutside = (e) => {
      if (!dashboard.contains(e.target) && !this.dashboardButton.contains(e.target)) {
        dashboard.remove();
        document.removeEventListener('click', closeOnOutside);
      }
    };
    setTimeout(() => document.addEventListener('click', closeOnOutside), 100);
  }

  // Helper functions

  detectPlatform() {
    const url = window.location.hostname;
    if (url.includes('chatgpt.com') || url.includes('chat.openai.com')) return 'ChatGPT';
    if (url.includes('claude.ai')) return 'Claude';
    if (url.includes('grok')) return 'Grok';
    if (url.includes('perplexity.ai')) return 'Perplexity';
    if (url.includes('gemini.google.com')) return 'Gemini';
    return 'Unknown';
  }

  // Extract visible conversation from current session for context-aware recall
  getSessionContext() {
    const platform = this.detectPlatform();

    try {
      // Platform-specific selectors for chat messages (more comprehensive)
      const selectors = {
        ChatGPT: '[data-message-author-role], .markdown, .text-base',
        Claude: '[class*="Message"], .prose, [class*="human"], [class*="assistant"]',
        Grok: '[class*="message"], [class*="Message"], [class*="chat"]',
        Perplexity: '[class*="prose"], [class*="answer"], [class*="response"]',
        Gemini: '[class*="response"], .message-content, [class*="model-response"]'
      };

      const selector = selectors[platform];
      if (!selector) return null;

      // Get all visible messages
      const messageElements = document.querySelectorAll(selector);

      // Extract text from last 8 messages for richer context
      const recentMessages = Array.from(messageElements).slice(-8);

      const messages = recentMessages
        .map((el) => {
          const text = el.textContent?.trim();
          if (!text || text.length < 15) return null;
          // Truncate very long messages but keep more content
          return text.length > 800 ? text.substring(0, 800) + '...' : text;
        })
        .filter(Boolean);

      if (messages.length === 0) return null;

      // Join messages with separator - allow up to 3000 chars for richer context
      return messages.join('\n---\n').substring(0, 3000);
    } catch (e) {
      console.log('[Haiven] Could not extract session context:', e.message);
      return null;
    }
  }

  getPlatformIcon(platform) {
    const icons = {
      ChatGPT: '🤖',
      Claude: '🧠',
      Grok: '✨',
      Perplexity: '🔍',
      Gemini: '💎',
      Unknown: '💭'
    };
    return icons[platform] || icons['Unknown'];
  }

  formatTimestamp(timestamp) {
    if (!timestamp) return 'recently';

    const date = new Date(timestamp);
    const now = new Date();
    const diffMs = now - date;
    const diffMins = Math.floor(diffMs / 60000);
    const diffHours = Math.floor(diffMs / 3600000);
    const diffDays = Math.floor(diffMs / 86400000);

    if (diffMins < 1) return 'just now';
    if (diffMins < 60) return `${diffMins} min${diffMins > 1 ? 's' : ''} ago`;
    if (diffHours < 24) return `${diffHours} hour${diffHours > 1 ? 's' : ''} ago`;
    if (diffDays < 7) return `${diffDays} day${diffDays > 1 ? 's' : ''} ago`;

    return date.toLocaleDateString();
  }

  getRecencyClass(timestamp) {
    if (!timestamp) return 'haiven-recent';

    const date = new Date(timestamp);
    const now = new Date();
    const diffHours = Math.floor((now - date) / 3600000);

    // Fresh: < 24 hours (green)
    if (diffHours < 24) return 'haiven-fresh';

    // Recent: 1-7 days (cyan)
    if (diffHours < 168) return 'haiven-recent';

    // Moderate: 1-4 weeks (orange)
    if (diffHours < 672) return 'haiven-moderate';

    // Old: > 4 weeks (gray)
    return 'haiven-old';
  }

  escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // Format single memory for clean injection
  formatMemoryForInjection(memory) {
    const platform = memory.metadata?.platform || 'Haiven';
    const timestamp = this.formatTimestamp(memory.created_at);

    return `📝 **Context from ${platform}** (${timestamp}):\n\n${memory.content}\n\n---\n\n`;
  }

  // Format multiple memories for clean injection
  formatMultipleMemoriesForInjection(memories) {
    const count = memories.length;
    const header = `📚 **Context from ${count} Haiven ${count > 1 ? 'Memories' : 'Memory'}:**\n\n`;

    const formattedMemories = memories
      .map((memory, index) => {
        const platform = memory.metadata?.platform || 'Haiven';
        const timestamp = this.formatTimestamp(memory.created_at);
        return `**Memory ${index + 1}** (${platform}, ${timestamp}):\n${memory.content}`;
      })
      .join('\n\n');

    return `${header}${formattedMemories}\n\n---\n\n`;
  }

  // ========================================
  // USER LEARNING - Track behavior to improve recall
  // ========================================

  async loadUserBehavior() {
    try {
      const data = await window.browserAPI.storage.local.get(['haivenUserBehavior']);
      if (data.haivenUserBehavior) {
        this.userBehavior = { ...this.userBehavior, ...data.haivenUserBehavior };
      }
      console.log('[Haiven] Loaded user behavior:', Object.keys(this.userBehavior));
    } catch (_e) {
      console.log('[Haiven] Could not load user behavior');
    }
  }

  async saveUserBehavior() {
    try {
      await window.browserAPI.storage.local.set({ haivenUserBehavior: this.userBehavior });
    } catch (_e) {
      console.log('[Haiven] Could not save user behavior');
    }
  }

  // Track when user actually uses a memory (positive signal)
  trackMemoryUsed(memory) {
    const memoryId = memory.id;
    if (!this.userBehavior.usedMemoryIds.includes(memoryId)) {
      this.userBehavior.usedMemoryIds.push(memoryId);
      // Keep only last 100
      if (this.userBehavior.usedMemoryIds.length > 100) {
        this.userBehavior.usedMemoryIds.shift();
      }
    }

    // Track platform usage
    const platform = this.detectPlatform();
    this.userBehavior.platformStats[platform] =
      (this.userBehavior.platformStats[platform] || 0) + 1;

    // Extract and track topics from memory content
    this.trackTopics(memory.content);

    // Send behavior to backend for personalization
    this.sendBehaviorSignal('used', memory);

    this.saveUserBehavior();
  }

  // Track when user dismisses without using (negative signal)
  trackDismissed(memories) {
    memories.forEach((memory) => {
      if (!this.userBehavior.dismissedMemoryIds.includes(memory.id)) {
        this.userBehavior.dismissedMemoryIds.push(memory.id);
        if (this.userBehavior.dismissedMemoryIds.length > 100) {
          this.userBehavior.dismissedMemoryIds.shift();
        }
      }
    });

    this.saveUserBehavior();
  }

  // Extract topics from content to build user profile
  trackTopics(content) {
    const lower = content.toLowerCase();
    const topics = [
      'code',
      'design',
      'business',
      'health',
      'finance',
      'learning',
      'writing',
      'marketing',
      'ai',
      'data',
      'product',
      'career'
    ];

    topics.forEach((topic) => {
      if (lower.includes(topic)) {
        this.userBehavior.topicAffinities[topic] =
          (this.userBehavior.topicAffinities[topic] || 0) + 1;
      }
    });
  }

  // Send behavior signal to backend for ML personalization
  async sendBehaviorSignal(action, memory) {
    try {
      const { userId } = await window.browserAPI.storage.local.get(['userId']);
      if (!userId) return;

      await fetch('https://api.safehaiven.com/api/memory/behavior', {
        method: 'POST',
        headers: await window.getAuthHeaders(),
        body: JSON.stringify({
          userId,
          action, // 'used' or 'dismissed'
          memoryId: memory.id,
          platform: this.detectPlatform(),
          context: this.lastQuery?.substring(0, 200),
          topicAffinities: this.userBehavior.topicAffinities
        })
      });
    } catch (_e) {
      // Silent fail - behavior tracking is optional
    }
  }

  // Get user's top topics for context enrichment
  getTopTopics(limit = 5) {
    const sorted = Object.entries(this.userBehavior.topicAffinities)
      .sort((a, b) => b[1] - a[1])
      .slice(0, limit);
    return sorted.map(([topic]) => topic);
  }
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', () => {
    new PredictiveMemoryOverlay();
  });
} else {
  new PredictiveMemoryOverlay();
}
