/* global browser, chrome */
// Browser compatibility layer
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
// SIMPLIFIED VERSION - Matches test script exactly

// All available Worlds (full catalog)
const ALL_WORLDS = {
  'Work & Career': '💼',
  'Marketing & Sales': '📈',
  'Legal & Compliance': '⚖️',
  'Food & Recipes': '🍳',
  'Technology & Coding': '💻',
  'Health & Fitness': '💪',
  'Finance & Money': '💰',
  'Creative & Design': '🎨',
  'Learning & Research': '📚',
  'Travel & Places': '✈️',
  'Family & Personal': '❤️',
  Entertainment: '🎬',
  'Business Strategy': '🎯',
  'Shopping & Products': '🛍️'
};

// ========================================
// AUTH HELPER - Get headers with API key or userId as fallback
// ========================================
async function getAuthHeaders() {
  const { apiKey, userId } = await browserAPI.storage.local.get(['apiKey', 'userId']);
  const headers = {
    'Content-Type': 'application/json'
  };
  // Use API key if available, otherwise use userId as Bearer token
  if (apiKey) {
    headers['Authorization'] = `Bearer ${apiKey}`;
  } else if (userId) {
    headers['Authorization'] = `Bearer ${userId}`;
  }
  return headers;
}

// ========================================
// CUSTOM MODAL SYSTEM - Replaces native confirm/alert
// ========================================
function showPopupModal({ title, message, type = 'info', buttons = ['OK'] }) {
  return new Promise((resolve) => {
    // Remove any existing modal
    document.getElementById('haiven-popup-modal')?.remove();

    const icons = { info: '💡', success: '✅', error: '❌', warning: '⚠️', confirm: '❓' };
    const colors = {
      info: 'var(--haiven-cyan)',
      success: 'var(--success-green)',
      error: 'var(--error-red)',
      warning: 'var(--warning-amber)',
      confirm: 'var(--haiven-cyan)'
    };

    const overlay = document.createElement('div');
    overlay.id = 'haiven-popup-modal';
    overlay.style.cssText = `
      position: fixed;
      inset: 0;
      background: rgba(0, 0, 0, 0.8);
      backdrop-filter: blur(4px);
      display: flex;
      align-items: center;
      justify-content: center;
      z-index: 10000;
      animation: fadeIn 0.2s ease-out;
    `;

    const modal = document.createElement('div');
    modal.style.cssText = `
      background: var(--bg-primary);
      border: 1px solid var(--border-medium);
      border-radius: var(--radius-lg);
      padding: 20px;
      max-width: 320px;
      width: 90%;
      box-shadow: var(--shadow-lg);
      animation: slideIn 0.3s ease-out;
    `;

    // SECURITY: Build modal with safe DOM methods instead of innerHTML
    // Header with icon and title
    const header = document.createElement('div');
    header.style.cssText = 'display: flex; align-items: center; gap: 12px; margin-bottom: 12px;';

    const iconSpan = document.createElement('span');
    iconSpan.style.fontSize = '24px';
    iconSpan.textContent = icons[type] || icons.info;
    header.appendChild(iconSpan);

    const titleH3 = document.createElement('h3');
    titleH3.style.cssText =
      'margin: 0; font-size: 16px; font-weight: 600; color: var(--text-primary);';
    titleH3.textContent = title; // SAFE: textContent
    header.appendChild(titleH3);

    // Message
    const messagePara = document.createElement('p');
    messagePara.style.cssText =
      'margin: 0 0 16px 0; font-size: 13px; color: var(--text-secondary); line-height: 1.5;';
    messagePara.textContent = message; // SAFE: textContent

    // Buttons container
    const buttonsDiv = document.createElement('div');
    buttonsDiv.style.cssText = 'display: flex; gap: 8px; justify-content: flex-end;';

    buttons.forEach((btnText, idx) => {
      const button = document.createElement('button');
      button.className = 'modal-btn';
      button.dataset.result = btnText;
      button.textContent = btnText; // SAFE: textContent
      button.style.cssText = `
        padding: 8px 16px;
        border-radius: var(--radius-sm);
        border: none;
        font-size: 13px;
        font-weight: 600;
        cursor: pointer;
        transition: all 0.2s;
        background: ${idx === buttons.length - 1 ? colors[type] : 'var(--bg-elevated)'};
        color: ${idx === buttons.length - 1 ? 'white' : 'var(--text-secondary)'};
      `;
      buttonsDiv.appendChild(button);
    });

    modal.appendChild(header);
    modal.appendChild(messagePara);
    modal.appendChild(buttonsDiv);

    overlay.appendChild(modal);
    document.body.appendChild(overlay);

    // Handle button clicks
    modal.querySelectorAll('.modal-btn').forEach((btn) => {
      btn.addEventListener('click', () => {
        overlay.remove();
        resolve(btn.dataset.result);
      });
    });

    // Close on overlay click
    overlay.addEventListener('click', (e) => {
      if (e.target === overlay) {
        overlay.remove();
        resolve(null);
      }
    });
  });
}

async function showConfirmDialog(title, message) {
  const result = await showPopupModal({
    title,
    message,
    type: 'confirm',
    buttons: ['Cancel', 'Delete']
  });
  return result === 'Delete';
}

document.addEventListener('DOMContentLoaded', async () => {
  // Check if user is authenticated
  const { userId } = await browserAPI.storage.local.get(['userId']);

  if (userId && userId.trim()) {
    // User is authenticated - show main view
    showMainView();
    await initializeWorldDropdown();
    await loadRecentMemories();
    document.getElementById('content').focus();
  } else {
    // New user - show onboarding
    showOnboardingView();
  }
});

// View switching functions
function showOnboardingView() {
  document.getElementById('onboardingView').classList.add('active');
  document.getElementById('mainView').classList.remove('active');
  document.getElementById('onboardingEmail').focus();
}

function showMainView() {
  document.getElementById('onboardingView').classList.remove('active');
  document.getElementById('mainView').classList.add('active');
}

// Onboarding button handler
document.getElementById('onboardingStartBtn').addEventListener('click', async () => {
  const emailInput = document.getElementById('onboardingEmail');
  const emailContainer = document.getElementById('emailContainer');
  const statusEl = document.getElementById('onboardingStatus');
  const btn = document.getElementById('onboardingStartBtn');
  const email = emailInput.value.trim().toLowerCase();

  // Reset states
  emailContainer.classList.remove('error', 'success');

  if (!email) {
    emailContainer.classList.add('error');
    statusEl.textContent = 'Please enter your email address'; // SECURITY: textContent prevents XSS
    statusEl.className = 'status error';
    emailInput.focus();
    return;
  }

  // Basic email validation
  if (!email.includes('@') || !email.includes('.')) {
    emailContainer.classList.add('error');
    statusEl.textContent = 'Please enter a valid email address'; // SECURITY: textContent
    statusEl.className = 'status error';
    emailInput.focus();
    return;
  }

  btn.disabled = true;
  btn.textContent = '⏳ Connecting...'; // SECURITY: textContent
  statusEl.textContent = '';

  try {
    // Look up user by email
    const response = await fetch('https://api.safehaiven.com/api/auth/lookup', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email })
    });

    if (response.ok) {
      const data = await response.json();
      if (data.userId) {
        // Found user - save and proceed
        await browserAPI.storage.local.set({
          userId: data.userId,
          userEmail: email
        });

        emailContainer.classList.add('success');
        statusEl.textContent = 'Connected successfully!'; // SECURITY: textContent
        statusEl.className = 'status success';

        setTimeout(async () => {
          showMainView();
          await initializeWorldDropdown();
          document.getElementById('content').focus();
          showStatus('Welcome! Start capturing memories', 'success');
        }, 1200);
      } else {
        throw new Error('User not found');
      }
    } else if (response.status === 404) {
      emailContainer.classList.add('error');
      statusEl.innerHTML =
        '<div class="status error">No account found with that email.<br><a href="https://www.safehaiven.com" target="_blank" style="color: var(--haiven-cyan); text-decoration: underline;">Create a free account</a></div>';
      btn.disabled = false;
      btn.textContent = '🔗 Connect Account'; // SECURITY: textContent
    } else {
      throw new Error('Connection failed');
    }
  } catch (error) {
    console.error('Onboarding error:', error);
    emailContainer.classList.add('error');
    statusEl.textContent = 'Could not connect. Please try again.'; // SECURITY: textContent
    statusEl.className = 'status error';
    btn.disabled = false;
    btn.textContent = '🔗 Connect Account';
  }
});

// Handle Enter key in onboarding input
document.getElementById('onboardingEmail').addEventListener('keydown', (e) => {
  if (e.key === 'Enter') {
    e.preventDefault();
    document.getElementById('onboardingStartBtn').click();
  }
});

// Capture button handler
document.getElementById('captureBtn').addEventListener('click', async () => {
  const content = document.getElementById('content').value.trim();
  const status = document.getElementById('status');
  const btn = document.getElementById('captureBtn');

  if (!content) {
    showStatus('⚠️ Please enter something to remember', 'error');
    return;
  }

  btn.disabled = true;
  showStatus('⏳ Saving memory...', 'info');

  try {
    // Get userId from storage
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    if (!userId) {
      showStatus('⚠️ Please set your User ID in settings first', 'error');
      btn.disabled = false;
      return;
    }

    // Get optional world and tags
    const world = document.getElementById('world').value;
    const tagsInput = document.getElementById('tags').value.trim();
    const tags = tagsInput
      ? tagsInput
          .split(',')
          .map((t) => t.trim())
          .filter((t) => t)
      : [];

    // EXACT SAME FORMAT AS TEST SCRIPT (which works!)
    const requestBody = {
      userId: userId,
      content: content,
      source: 'chrome-extension',
      tags: tags,
      world_name: world || null
    };

    console.log('📤 Sending request...');
    console.log('URL:', 'https://api.safehaiven.com/api/memory/store/fast');
    console.log('Body:', requestBody);

    const response = await fetch('https://api.safehaiven.com/api/memory/store/fast', {
      method: 'POST',
      headers: await getAuthHeaders(),
      body: JSON.stringify(requestBody)
    });

    console.log('📥 Response status:', response.status);

    if (!response.ok) {
      const errorText = await response.text();
      console.error('❌ Response text:', errorText);

      try {
        const errorData = JSON.parse(errorText);
        throw new Error(
          errorData.error || errorData.details || `Server error (${response.status})`
        );
      } catch (_e) {
        throw new Error(`Server returned ${response.status}: ${errorText.substring(0, 100)}`);
      }
    }

    const data = await response.json();
    console.log('✅ Response data:', data);

    if (data.success) {
      // 🔄 CHECK FOR DEDUPLICATION: Memory already existed
      if (data.action === 'deduplicated') {
        console.log('🔄 Memory deduplicated - already exists:', data.existingMemoryId);
        showStatus('🔄 You already have this memory! Importance updated.', 'info');

        // Clear form after short delay so user sees the message
        setTimeout(() => {
          document.getElementById('content').value = '';
          document.getElementById('tags').value = '';
        }, 1500);

        // Refresh recent memories list to show existing memory
        await loadRecentMemories();

        // Clear status after 4 seconds
        setTimeout(() => {
          status.className = '';
          status.textContent = '';
        }, 4000);
      } else {
        // New memory saved
        showStatus('✅ Memory saved successfully!', 'success');
        console.log('✅ Memory ID:', data.memory.id);

        // 🆕 PHASE 2: Display detected emotions
        if (data.memory && data.memory.metadata && data.memory.metadata.emotions) {
          displayEmotions(data.memory.metadata.emotions);
        }

        // Clear form
        document.getElementById('content').value = '';
        document.getElementById('tags').value = '';

        // Refresh recent memories list
        await loadRecentMemories();

        // Clear status after 3 seconds (give more time to see emotions)
        setTimeout(() => {
          status.className = '';
          status.textContent = '';
        }, 3000);
      }
    } else if (data.rejected) {
      // Content was filtered - show helpful message
      console.log('⚠️ Memory rejected:', data.reason);
      showStatus(
        `⚠️ Not saved: ${data.reason || 'Content filtered as not memorable'}. Try adding more personal context or specific details.`,
        'warning'
      );
      // Don't clear form so user can edit and retry
    } else {
      throw new Error(data.error || 'Failed to save');
    }
  } catch (error) {
    console.error('❌ Save error:', error);
    showStatus(`❌ Error: ${error.message}`, 'error');
  } finally {
    btn.disabled = false;
  }
});

// Open dashboard with auto-login
document.getElementById('openDashboard').addEventListener('click', async () => {
  try {
    const { userId } = await browserAPI.storage.local.get(['userId']);

    if (userId) {
      // Get a one-time login token
      const response = await fetch('https://api.safehaiven.com/api/auth/extension-token', {
        method: 'POST',
        headers: await getAuthHeaders(),
        body: JSON.stringify({ userId })
      });

      if (response.ok) {
        const data = await response.json();
        if (data.token) {
          // Open dashboard with auto-login token
          browserAPI.tabs.create({
            url: `https://www.safehaiven.com/dashboard.html?ext_token=${data.token}`
          });
          return;
        }
      }
    }

    // Fallback: open dashboard without auto-login
    browserAPI.tabs.create({ url: 'https://www.safehaiven.com/dashboard.html' });
  } catch (error) {
    console.error('Dashboard auto-login failed:', error);
    browserAPI.tabs.create({ url: 'https://www.safehaiven.com/dashboard.html' });
  }
});

// View conversations/threads
document.getElementById('viewThreads').addEventListener('click', () => {
  browserAPI.tabs.create({ url: browserAPI.runtime.getURL('threads.html') });
});

// View recent decisions
document.getElementById('viewDecisions')?.addEventListener('click', () => {
  browserAPI.tabs.create({ url: 'https://www.safehaiven.com/dashboard.html?view=decisions' });
});

// SEARCH FUNCTIONALITY (the new hero action)
document.getElementById('searchBtn')?.addEventListener('click', async () => {
  await performSearch();
});

document.getElementById('searchInput')?.addEventListener('keydown', (e) => {
  if (e.key === 'Enter') {
    e.preventDefault();
    performSearch();
  }
});

async function performSearch() {
  const searchInput = document.getElementById('searchInput');
  const searchResults = document.getElementById('searchResults');
  const searchBtn = document.getElementById('searchBtn');
  const query = searchInput?.value?.trim();

  if (!query) {
    showStatus('Enter a search query', 'info');
    return;
  }

  searchBtn.disabled = true;
  searchBtn.innerHTML = '<span>🔍</span><span>Searching...</span>';
  searchResults.style.display = 'block';
  searchResults.innerHTML = '<div class="loading">Searching your memories...</div>';

  try {
    const { userId } = await browserAPI.storage.local.get(['userId']);
    if (!userId) {
      searchResults.innerHTML = '<div class="status error">Not logged in</div>';
      return;
    }

    const headers = await getAuthHeaders();
    const response = await fetch(
      `https://api.safehaiven.com/api/memory/search?userId=${userId}&q=${encodeURIComponent(query)}&limit=5`,
      {
        method: 'GET',
        headers
      }
    );

    if (!response.ok) throw new Error('Search failed');

    const data = await response.json();
    const memories = data.memories || data.results || [];

    if (memories.length === 0) {
      searchResults.innerHTML = `
        <div class="empty-state" style="padding: 20px; text-align: center;">
          <div style="font-size: 32px; margin-bottom: 8px;">🔍</div>
          <div style="color: var(--text-secondary); font-size: 13px;">No memories found for "${query}"</div>
          <div style="color: var(--text-tertiary); font-size: 12px; margin-top: 4px;">Try different keywords</div>
        </div>
      `;
    } else {
      searchResults.innerHTML = memories
        .map(
          (m) => `
        <div class="memory-card" style="margin-bottom: 8px;">
          <div class="memory-content">${escapeHtml(m.content || m.text || '')}</div>
          <div class="memory-meta">
            <span class="memory-time">${formatRelativeTime(m.created_at || m.createdAt)}</span>
            ${m.type ? `<span class="memory-tag">${m.type}</span>` : ''}
          </div>
        </div>
      `
        )
        .join('');
    }
  } catch (error) {
    console.error('Search error:', error);
    searchResults.innerHTML = '<div class="status error">Search failed. Try again.</div>';
  } finally {
    searchBtn.disabled = false;
    searchBtn.innerHTML = '<span>🔍</span><span>Search memories</span>';
  }
}

function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

function formatRelativeTime(dateStr) {
  if (!dateStr) return '';
  const date = new Date(dateStr);
  const now = new Date();
  const diffMs = now - date;
  const diffMins = Math.floor(diffMs / 60000);
  const diffHours = Math.floor(diffMins / 60);
  const diffDays = Math.floor(diffHours / 24);

  if (diffMins < 1) return 'just now';
  if (diffMins < 60) return `${diffMins}m ago`;
  if (diffHours < 24) return `${diffHours}h ago`;
  if (diffDays < 7) return `${diffDays}d ago`;
  return date.toLocaleDateString();
}

// Capture selection - Simplified (moved to collapsible section)

// Settings
document.getElementById('openSettings').addEventListener('click', () => {
  browserAPI.runtime.openOptionsPage();
});

// Keyboard shortcut
document.getElementById('content').addEventListener('keydown', (e) => {
  if ((e.ctrlKey || e.metaKey) && e.key === 'Enter') {
    e.preventDefault();
    document.getElementById('captureBtn').click();
  }
});

// Helper
function showStatus(message, type = 'info') {
  const status = document.getElementById('status');
  status.textContent = message;
  status.className = `status ${type}`;
}

// 🆕 PHASE 2: Display emotions with beautiful badges
function displayEmotions(emotions) {
  const emotionDisplay = document.getElementById('emotionDisplay');
  const emotionBadges = document.getElementById('emotionBadges');
  const emotionState = document.getElementById('emotionState');

  if (!emotions || !emotions.detected || emotions.detected.length === 0) {
    emotionDisplay.style.display = 'none';
    return;
  }

  // Clear previous
  emotionBadges.textContent = ''; // SECURITY: textContent (empty reset)
  emotionState.textContent = '';

  // Show container
  emotionDisplay.style.display = 'block';

  // Create emotion badges
  const primary = emotions.primary || emotions.detected.slice(0, 1);
  const secondary = emotions.secondary || emotions.detected.slice(1);

  // Add primary emotions
  primary.forEach((emotion) => {
    const badge = createEmotionBadge(emotion, emotions.valence, true);
    emotionBadges.appendChild(badge);
  });

  // Add secondary emotions
  secondary.forEach((emotion) => {
    const badge = createEmotionBadge(emotion, emotions.valence, false);
    emotionBadges.appendChild(badge);
  });

  // Show emotional state
  if (emotions.emotionalState) {
    emotionState.textContent = `"${emotions.emotionalState}"`;
  }

  // Auto-hide after 10 seconds
  setTimeout(() => {
    emotionDisplay.style.display = 'none';
  }, 10000);
}

function createEmotionBadge(emotion, valence, isPrimary) {
  const badge = document.createElement('div');

  // Get emoji for emotion
  const emoji = getEmotionEmoji(emotion);

  // Get color based on valence
  let bgColor, borderColor;
  if (valence > 0.3) {
    bgColor = 'rgba(16, 185, 129, 0.15)'; // Green
    borderColor = 'rgba(16, 185, 129, 0.3)';
  } else if (valence < -0.3) {
    bgColor = 'rgba(239, 68, 68, 0.15)'; // Red
    borderColor = 'rgba(239, 68, 68, 0.3)';
  } else {
    bgColor = 'rgba(59, 130, 246, 0.15)'; // Blue
    borderColor = 'rgba(59, 130, 246, 0.3)';
  }

  badge.style.cssText = `
    display: inline-flex;
    align-items: center;
    gap: 4px;
    padding: 4px 10px;
    background: ${bgColor};
    border: 1px solid ${borderColor};
    border-radius: 12px;
    font-size: 11px;
    font-weight: ${isPrimary ? '600' : '500'};
    opacity: ${isPrimary ? '1' : '0.8'};
  `;

  // SECURITY: Build badge with safe DOM
  const emojiSpan = document.createElement('span');
  emojiSpan.textContent = emoji;

  const textSpan = document.createElement('span');
  textSpan.textContent = emotion; // SAFE: API data via textContent

  badge.appendChild(emojiSpan);
  badge.appendChild(textSpan);

  return badge;
}

function getEmotionEmoji(emotion) {
  const emojiMap = {
    anxious: '😰',
    happy: '😊',
    excited: '🤩',
    sad: '😢',
    frustrated: '😤',
    proud: '💪',
    grateful: '🙏',
    stressed: '😓',
    hopeful: '🤞',
    worried: '😟',
    relieved: '😌',
    angry: '😠',
    loving: '❤️',
    confident: '😎',
    scared: '😨',
    disappointed: '😞',
    embarrassed: '😳',
    overwhelmed: '😵',
    confused: '😕',
    lonely: '😔',
    ashamed: '😔',
    guilty: '😔',
    nervous: '😬',
    amazed: '😲',
    surprised: '😮',
    content: '😌',
    calm: '😌',
    hurt: '😔',
    jealous: '😒',
    bitter: '😑'
  };

  return emojiMap[emotion] || '💭';
}

// ============================================
// RECENT MEMORIES FUNCTIONALITY
// ============================================

let _recentMemories = []; // Reserved for future use

// Refresh button handler
document.getElementById('refreshMemories').addEventListener('click', async () => {
  await loadRecentMemories();
});

/**
 * Fetch and display recent memories
 */
async function loadRecentMemories() {
  const memoriesList = document.getElementById('memoriesList');
  memoriesList.textContent = 'Loading recent memories...'; // SECURITY: textContent
  memoriesList.className = 'loading';

  try {
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    if (!userId) {
      memoriesList.textContent = '⚙️ Please set your User ID in settings first'; // SECURITY: textContent
      memoriesList.className = 'no-memories';
      return;
    }

    // Fetch recent memories from API
    const response = await fetch(`https://api.safehaiven.com/api/memory/recent/${userId}?limit=5`, {
      method: 'GET',
      headers: await getAuthHeaders()
    });

    if (!response.ok) {
      throw new Error(`API returned ${response.status}`);
    }

    const data = await response.json();

    if (data.success && data.memories && data.memories.length > 0) {
      _recentMemories = data.memories;
      renderMemories(data.memories);
    } else {
      memoriesList.textContent = '📭 No memories yet. Start capturing!'; // SECURITY: textContent
      memoriesList.className = 'no-memories';
    }
  } catch (error) {
    console.error('Failed to load memories:', error);
    memoriesList.textContent = '⚠️ Failed to load memories'; // SECURITY: textContent
    memoriesList.className = 'no-memories';
  }
}

/**
 * Render memories list
 */
function renderMemories(memories) {
  const memoriesList = document.getElementById('memoriesList');
  memoriesList.textContent = ''; // SECURITY: textContent (clearing)

  memories.forEach((memory) => {
    const card = createMemoryCard(memory);
    memoriesList.appendChild(card);
  });
}

/**
 * Create memory card element
 */
function createMemoryCard(memory) {
  const card = document.createElement('div');
  card.className = 'memory-card';
  card.dataset.memoryId = memory.id;

  // Memory content
  const content = document.createElement('div');
  content.className = 'memory-content';
  content.textContent = memory.content;
  card.appendChild(content);

  // World badge (if exists)
  if (memory.world_name) {
    const worldBadge = document.createElement('div');
    worldBadge.className = 'memory-world';
    worldBadge.style.cssText =
      'display: inline-block; padding: 4px 8px; margin-top: 8px; margin-bottom: 4px; background: rgba(139,92,246,0.2); border: 1px solid rgba(139,92,246,0.4); border-radius: 6px; font-size: 11px; font-weight: 600; color: var(--neural-purple);';
    worldBadge.textContent = `🌍 ${memory.world_name}`;
    card.appendChild(worldBadge);
  }

  // Tags
  if (memory.tags && memory.tags.length > 0) {
    const tagsContainer = document.createElement('div');
    tagsContainer.className = 'memory-tags';
    memory.tags.forEach((tag) => {
      const tagEl = document.createElement('span');
      tagEl.className = 'memory-tag';
      tagEl.textContent = tag;
      tagsContainer.appendChild(tagEl);
    });
    card.appendChild(tagsContainer);
  }

  // Meta (time + actions)
  const meta = document.createElement('div');
  meta.className = 'memory-meta';

  // Time
  const timeEl = document.createElement('div');
  timeEl.className = 'memory-time';
  // SECURITY: Build time element safely
  const clockSpan = document.createElement('span');
  clockSpan.textContent = '🕐';

  const timeTextSpan = document.createElement('span');
  timeTextSpan.textContent = formatTimeAgo(memory.created_at); // SAFE: formatted timestamp

  timeEl.appendChild(clockSpan);
  timeEl.appendChild(timeTextSpan);
  meta.appendChild(timeEl);

  // Actions
  const actions = document.createElement('div');
  actions.className = 'memory-actions';

  // Edit button
  const editBtn = document.createElement('button');
  editBtn.className = 'memory-action-btn';
  editBtn.textContent = '✏️'; // SECURITY: textContent
  editBtn.title = 'Edit';
  editBtn.onclick = () => editMemory(memory);
  actions.appendChild(editBtn);

  // Tag button
  const tagBtn = document.createElement('button');
  tagBtn.className = 'memory-action-btn';
  tagBtn.textContent = '🏷️'; // SECURITY: textContent
  tagBtn.title = 'Add Tags';
  tagBtn.onclick = () => addTags(memory);
  actions.appendChild(tagBtn);

  // Delete button
  const deleteBtn = document.createElement('button');
  deleteBtn.className = 'memory-action-btn';
  deleteBtn.textContent = '🗑️'; // SECURITY: textContent
  deleteBtn.title = 'Delete';
  deleteBtn.onclick = () => deleteMemory(memory);
  actions.appendChild(deleteBtn);

  meta.appendChild(actions);
  card.appendChild(meta);

  return card;
}

/**
 * Format time ago (e.g., "2 minutes ago")
 */
function formatTimeAgo(timestamp) {
  const now = new Date();
  const past = new Date(timestamp);
  const diff = now - past;

  const seconds = Math.floor(diff / 1000);
  const minutes = Math.floor(seconds / 60);
  const hours = Math.floor(minutes / 60);
  const days = Math.floor(hours / 24);

  if (seconds < 60) return 'just now';
  if (minutes < 60) return `${minutes}m ago`;
  if (hours < 24) return `${hours}h ago`;
  if (days < 7) return `${days}d ago`;
  return past.toLocaleDateString();
}

/**
 * Edit memory
 */
async function editMemory(memory) {
  const newContent = prompt('Edit memory:', memory.content);

  if (newContent === null || newContent.trim() === '') {
    return; // User cancelled or empty
  }

  if (newContent === memory.content) {
    return; // No changes
  }

  try {
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    const response = await fetch(`https://api.safehaiven.com/api/memory/update/${memory.id}`, {
      method: 'PUT',
      headers: await getAuthHeaders(),
      body: JSON.stringify({
        userId: userId,
        content: newContent.trim()
      })
    });

    const data = await response.json();

    if (data.success) {
      showStatus('✅ Memory updated!', 'success');
      await loadRecentMemories(); // Refresh list
    } else {
      throw new Error(data.error || 'Update failed');
    }
  } catch (error) {
    console.error('Edit error:', error);
    showStatus('❌ Failed to update memory', 'error');
  }
}

/**
 * Add/edit tags
 */
async function addTags(memory) {
  const currentTags = memory.tags ? memory.tags.join(', ') : '';
  const newTags = prompt('Enter tags (comma-separated):', currentTags);

  if (newTags === null) {
    return; // User cancelled
  }

  const tagsArray = newTags
    .split(',')
    .map((tag) => tag.trim())
    .filter((tag) => tag.length > 0);

  try {
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    const response = await fetch(`https://api.safehaiven.com/api/memory/update/${memory.id}`, {
      method: 'PUT',
      headers: await getAuthHeaders(),
      body: JSON.stringify({
        userId: userId,
        tags: tagsArray
      })
    });

    const data = await response.json();

    if (data.success) {
      showStatus('✅ Tags updated!', 'success');
      await loadRecentMemories(); // Refresh list
    } else {
      throw new Error(data.error || 'Update failed');
    }
  } catch (error) {
    console.error('Tag error:', error);
    showStatus('❌ Failed to update tags', 'error');
  }
}

/**
 * Delete memory
 */
async function deleteMemory(memory) {
  const preview =
    memory.content.length > 80 ? memory.content.substring(0, 80) + '...' : memory.content;
  const confirmed = await showConfirmDialog('Delete Memory?', `"${preview}"`);

  if (!confirmed) {
    return;
  }

  try {
    const result = await browserAPI.storage.local.get(['userId']);
    const userId = result.userId || '';

    const response = await fetch(`https://api.safehaiven.com/api/memory/delete/${memory.id}`, {
      method: 'DELETE',
      headers: await getAuthHeaders(),
      body: JSON.stringify({
        userId: userId
      })
    });

    const data = await response.json();

    if (data.success) {
      showStatus('✅ Memory deleted', 'success');
      await loadRecentMemories(); // Refresh list
    } else {
      throw new Error(data.error || 'Delete failed');
    }
  } catch (error) {
    console.error('Delete error:', error);
    showStatus('❌ Failed to delete memory', 'error');
  }
}

/**
 * Initialize smart World dropdown
 * - Shows only Worlds the user has used
 * - Restores last selected World (sticky)
 * - Allows adding new Worlds
 */
async function initializeWorldDropdown() {
  const worldSelect = document.getElementById('world');

  try {
    const result = await browserAPI.storage.local.get(['userId', 'lastWorld']);
    const userId = result.userId || '';

    if (!userId) {
      // No user, show simple picker
      populateWorldPicker(worldSelect, [], null);
      return;
    }

    // Fetch user's memories to extract unique Worlds
    const response = await fetch(
      `https://api.safehaiven.com/api/memory/recent/${userId}?limit=100`,
      { headers: await getAuthHeaders() }
    );

    if (!response.ok) {
      populateWorldPicker(worldSelect, [], result.lastWorld);
      return;
    }

    const data = await response.json();
    const memories = data.memories || [];

    // Extract unique Worlds from user's memories
    const userWorlds = [
      ...new Set(memories.map((m) => m.world_name).filter((w) => w && ALL_WORLDS[w]))
    ];

    // Populate dropdown with user's Worlds + option to add new
    populateWorldPicker(worldSelect, userWorlds, result.lastWorld);
  } catch (error) {
    console.error('Failed to initialize Worlds:', error);
    populateWorldPicker(worldSelect, [], null);
  }
}

/**
 * Populate World picker
 */
function populateWorldPicker(selectEl, userWorlds, lastWorld) {
  // Clear existing options except the first one
  // SECURITY: Build world selector safely
  const defaultOption = document.createElement('option');
  defaultOption.value = '';
  defaultOption.textContent = '🌍 Select a World (optional)';
  selectEl.appendChild(defaultOption);

  // Add user's Worlds first
  userWorlds.forEach((worldName) => {
    const option = document.createElement('option');
    option.value = worldName;
    option.textContent = `${ALL_WORLDS[worldName]} ${worldName}`;
    selectEl.appendChild(option);
  });

  // Add separator if user has Worlds
  if (userWorlds.length > 0) {
    const separator = document.createElement('option');
    separator.disabled = true;
    separator.textContent = '──────────';
    selectEl.appendChild(separator);
  }

  // Add all Worlds that user hasn't used yet
  Object.entries(ALL_WORLDS).forEach(([worldName, icon]) => {
    if (!userWorlds.includes(worldName)) {
      const option = document.createElement('option');
      option.value = worldName;
      option.textContent = `${icon} ${worldName}`;
      selectEl.appendChild(option);
    }
  });

  // Restore last selected World (sticky)
  if (lastWorld && ALL_WORLDS[lastWorld]) {
    selectEl.value = lastWorld;
  }

  // Save selection when changed (make it sticky)
  selectEl.addEventListener('change', () => {
    browserAPI.storage.local.set({ lastWorld: selectEl.value });
  });
}
