/* global browser, chrome, AbortController */
// Haiven Memory - Cross-Platform AI Companion
// Works across ChatGPT, Claude, Grok, Perplexity, Gemini, etc.
// Supports Chrome, Firefox, and Edge

(function () {
  'use strict';

  // ========================================
  // BROWSER COMPATIBILITY LAYER
  // ========================================
  const browserAPI = typeof browser !== 'undefined' ? browser : chrome;

  // ========================================
  // AUTH HELPER - Get headers with API key or userId as fallback
  // ========================================
  async function getAuthHeaders() {
    const { apiKey, userId } = await browserAPI.storage.local.get(['apiKey', 'userId']);
    const headers = {
      'Content-Type': 'application/json'
    };
    // Use API key if available, otherwise use userId as Bearer token
    if (apiKey) {
      headers['Authorization'] = `Bearer ${apiKey}`;
    } else if (userId) {
      headers['Authorization'] = `Bearer ${userId}`;
    }
    return headers;
  }

  // ========================================
  // SAFE FETCH WITH TIMEOUT - Prevents hanging requests
  // ========================================
  async function safeFetch(url, options = {}, timeoutMs = 10000) {
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), timeoutMs);

    try {
      const response = await fetch(url, {
        ...options,
        signal: controller.signal
      });
      return response;
    } catch (error) {
      if (error.name === 'AbortError') {
        throw new Error(`Request timed out after ${timeoutMs}ms`);
      }
      throw error;
    } finally {
      clearTimeout(timeoutId);
    }
  }

  try {
    // Version control - synced with manifest
    const EXTENSION_VERSION = browserAPI.runtime.getManifest().version;
    const CONFIDENCE_THRESHOLD = 0.75;
    const DEBOUNCE_DELAY_MS = 1500;
    const MIN_INPUT_LENGTH = 25;
    const MAX_REQUESTS_PER_MINUTE = 10;

    // ========================================
    // CLEANUP MANAGER - Prevents memory leaks
    // ========================================
    const cleanupManager = {
      intervals: [],
      observers: [],
      listeners: [],

      addInterval(id) {
        this.intervals.push(id);
        return id;
      },

      addObserver(observer) {
        this.observers.push(observer);
        return observer;
      },

      addListener(element, event, handler, options) {
        element.addEventListener(event, handler, options);
        this.listeners.push({ element, event, handler, options });
      },

      cleanup() {
        this.intervals.forEach((id) => clearInterval(id));
        this.observers.forEach((obs) => obs.disconnect());
        this.listeners.forEach(({ element, event, handler, options }) => {
          element.removeEventListener(event, handler, options);
        });
        console.log('🧹 Haiven: Cleaned up resources');
      }
    };

    // Cleanup on page unload
    window.addEventListener('beforeunload', () => cleanupManager.cleanup());
    window.addEventListener('pagehide', () => cleanupManager.cleanup());

    // ========================================
    // SAFE DOM UTILITIES - Prevents XSS
    // ========================================
    function createElement(tag, attrs = {}, children = []) {
      const el = document.createElement(tag);
      Object.entries(attrs).forEach(([key, value]) => {
        if (key === 'style' && typeof value === 'object') {
          Object.assign(el.style, value);
        } else if (key === 'className') {
          el.className = value;
        } else if (key === 'textContent') {
          el.textContent = value;
        } else if (key.startsWith('on') && typeof value === 'function') {
          el.addEventListener(key.slice(2).toLowerCase(), value);
        } else if (key === 'dataset') {
          Object.entries(value).forEach(([k, v]) => (el.dataset[k] = v));
        } else {
          el.setAttribute(key, value);
        }
      });
      children.forEach((child) => {
        if (typeof child === 'string') {
          el.appendChild(document.createTextNode(child));
        } else if (child) {
          el.appendChild(child);
        }
      });
      return el;
    }

    // ========================================
    // CUSTOM MODAL SYSTEM - Replaces alert/confirm
    // ========================================
    function showModal({ title, message, type = 'info', buttons = ['OK'], onResult }) {
      const overlay = createElement('div', {
        id: 'haiven-modal-overlay',
        style: {
          position: 'fixed',
          inset: '0',
          background: 'rgba(0, 0, 0, 0.7)',
          backdropFilter: 'blur(4px)',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          zIndex: '1000000',
          animation: 'haivenFadeIn 0.2s ease-out'
        }
      });

      const icons = { info: '💡', success: '✅', error: '❌', warning: '⚠️', confirm: '❓' };
      const colors = {
        info: '#f59e0b',
        success: '#10b981',
        error: '#ef4444',
        warning: '#f59e0b',
        confirm: '#f59e0b'
      };

      const modal = createElement(
        'div',
        {
          style: {
            background: '#1a1a1a',
            borderRadius: '16px',
            padding: '24px',
            maxWidth: '400px',
            width: '90%',
            boxShadow: `0 8px 32px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.1)`,
            animation: 'haivenSlideUp 0.3s ease-out'
          }
        },
        [
          createElement(
            'div',
            {
              style: {
                display: 'flex',
                alignItems: 'center',
                gap: '12px',
                marginBottom: '16px'
              }
            },
            [
              createElement('span', { style: { fontSize: '28px' } }, [icons[type] || icons.info]),
              createElement(
                'h3',
                {
                  style: {
                    margin: '0',
                    fontSize: '18px',
                    fontWeight: '600',
                    color: '#fff'
                  }
                },
                [title]
              )
            ]
          ),
          createElement(
            'p',
            {
              style: {
                margin: '0 0 20px 0',
                fontSize: '14px',
                color: 'rgba(255, 255, 255, 0.7)',
                lineHeight: '1.5'
              }
            },
            [message]
          ),
          createElement(
            'div',
            { style: { display: 'flex', gap: '10px', justifyContent: 'flex-end' } },
            buttons.map((btnText, idx) =>
              createElement(
                'button',
                {
                  style: {
                    padding: '10px 20px',
                    borderRadius: '8px',
                    border: 'none',
                    fontSize: '14px',
                    fontWeight: '600',
                    cursor: 'pointer',
                    transition: 'all 0.2s',
                    background: idx === buttons.length - 1 ? colors[type] : 'rgba(255,255,255,0.1)',
                    color: '#fff'
                  },
                  onClick: () => {
                    overlay.remove();
                    onResult?.(btnText);
                  }
                },
                [btnText]
              )
            )
          )
        ]
      );

      overlay.appendChild(modal);
      overlay.addEventListener('click', (e) => {
        if (e.target === overlay) {
          overlay.remove();
          onResult?.(null);
        }
      });

      // Add animation styles if not present
      if (!document.getElementById('haiven-modal-styles')) {
        const style = createElement('style', { id: 'haiven-modal-styles' });
        style.textContent = `
          @keyframes haivenFadeIn { from { opacity: 0; } to { opacity: 1; } }
          @keyframes haivenSlideUp { from { opacity: 0; transform: translateY(20px); } to { opacity: 1; transform: translateY(0); } }
          @keyframes haivenPanelEntrance {
            0% { opacity: 0; transform: scale(0.92) translateY(20px); }
            60% { transform: scale(1.02) translateY(-4px); }
            100% { opacity: 1; transform: scale(1) translateY(0); }
          }
          @keyframes haivenIconPulse { 0%, 100% { transform: scale(1); } 50% { transform: scale(1.1); } }
          @keyframes haivenStatusPulse { 0%, 100% { opacity: 1; box-shadow: 0 0 0 0 rgba(16, 185, 129, 0.4); } 50% { opacity: 0.8; box-shadow: 0 0 0 4px rgba(16, 185, 129, 0); } }
          @keyframes haivenShimmer { 0% { transform: translateX(-100%); } 100% { transform: translateX(100%); } }
          @keyframes haivenSpinLoader { from { transform: rotate(0deg); } to { transform: rotate(360deg); } }
          @keyframes haivenGlow { 0%, 100% { opacity: 0.4; } 50% { opacity: 0.6; } }
          @keyframes haivenFloat { 0%, 100% { transform: translateY(0); } 50% { transform: translateY(-2px); } }
          @keyframes haivenRingPulse { 0% { transform: scale(1); opacity: 1; } 100% { transform: scale(2); opacity: 0; } }
        `;
        document.head.appendChild(style);
      }

      document.body.appendChild(overlay);
      return overlay;
    }

    // Promise-based confirm dialog (reserved for future use)
    function _showConfirm(title, message) {
      return new Promise((resolve) => {
        showModal({
          title,
          message,
          type: 'confirm',
          buttons: ['Cancel', 'Confirm'],
          onResult: (result) => resolve(result === 'Confirm')
        });
      });
    }

    // Request throttling
    let requestCount = 0;
    let requestWindowStart = Date.now();

    // P0.3: Conversation context tracking
    let lastUserMessage = null;
    const _conversationPairs = []; // Track question-answer pairs (reserved for future use)

    console.log(`🚀 Haiven Extension v${EXTENSION_VERSION} - COST OPTIMIZED MODE`);
    console.log(
      `📍 Threshold: ${CONFIDENCE_THRESHOLD * 100}% | Debounce: ${DEBOUNCE_DELAY_MS}ms | Min length: ${MIN_INPUT_LENGTH} | Max req/min: ${MAX_REQUESTS_PER_MINUTE}`
    );

    // Platform detection
    const PLATFORMS = {
      CHATGPT: {
        name: 'ChatGPT',
        domains: ['chat.openai.com', 'chatgpt.com'],
        inputSelector: '#prompt-textarea, [data-testid="input-box"]',
        messageSelector: '[data-message-author-role="user"]',
        assistantSelector: '[data-message-author-role="assistant"]',
        composerSelector:
          'form[class*="composer"], [data-testid="composer"], .composer-parent, #prompt-textarea',
        sendButtonSelector:
          'button[data-testid="send-button"], button[aria-label*="Send"], form button[type="submit"]',
        promptStyle: 'conversational',
        maxTokens: 128000,
        supportsSystemPrompt: true
      },
      CLAUDE: {
        name: 'Claude',
        domains: ['claude.ai'],
        inputSelector: '[contenteditable="true"][data-placeholder]',
        messageSelector: '.font-user-message',
        assistantSelector: '.font-claude-message',
        composerSelector: '[class*="ProseMirror"], [contenteditable="true"]',
        sendButtonSelector: 'button[aria-label*="Send"], button[type="submit"]',
        promptStyle: 'xml-structured',
        maxTokens: 200000,
        supportsSystemPrompt: true
      },
      GROK: {
        name: 'Grok',
        domains: ['x.com', 'twitter.com', 'grok.com'],
        inputSelector:
          '[data-testid="grok-input"], [data-testid="chat-input"], textarea[placeholder*="Ask"], textarea[placeholder*="ask"], textarea[placeholder*="Message"], .composer textarea, [contenteditable="true"]',
        messageSelector: '[data-testid="grok-user-message"]',
        assistantSelector: '[data-testid="grok-assistant-message"]',
        composerSelector: '[data-testid="grok-composer"], [data-testid="grokComposer"], .composer',
        sendButtonSelector: '[data-testid="grok-send-button"], button[aria-label*="Send"]',
        promptStyle: 'direct',
        maxTokens: 25000,
        supportsSystemPrompt: false
      },
      PERPLEXITY: {
        name: 'Perplexity',
        domains: ['perplexity.ai'],
        inputSelector: 'textarea[placeholder*="Ask"]',
        messageSelector: '[data-role="user"]',
        assistantSelector: '[data-role="assistant"]',
        composerSelector: 'form, .search-input-container',
        sendButtonSelector: 'button[type="submit"], button[aria-label*="Search"]',
        promptStyle: 'research-focused',
        maxTokens: 16000,
        supportsSystemPrompt: false
      },
      GEMINI: {
        name: 'Gemini',
        domains: ['gemini.google.com'],
        inputSelector: '.ql-editor[contenteditable="true"]',
        messageSelector: '.user-message',
        assistantSelector: '.model-message',
        composerSelector: '.input-area, .ql-editor',
        sendButtonSelector: 'button[aria-label*="Send"], button[data-mat-icon-type]',
        promptStyle: 'conversational',
        maxTokens: 1000000,
        supportsSystemPrompt: true
      }
    };

    // Detect current platform
    function detectPlatform() {
      const { hostname } = window.location;
      for (const [key, platform] of Object.entries(PLATFORMS)) {
        if (platform.domains.some((domain) => hostname.includes(domain))) {
          return { key, ...platform };
        }
      }
      return null;
    }

    const currentPlatform = detectPlatform();
    if (!currentPlatform) {
      console.log('🧠 Haiven: Not on a supported AI platform');
      return;
    }

    console.log(`🧠 Haiven: Detected ${currentPlatform.name}`);

    // API Configuration - Initialize async to prevent race conditions
    let API_BASE = 'https://api.safehaiven.com';
    let USER_ID = '';

    // Load settings from storage BEFORE initializing extension
    async function loadSettings() {
      try {
        const result = await browserAPI.storage.local.get(['apiEndpoint', 'userId']);
        API_BASE = result.apiEndpoint || 'https://api.safehaiven.com';
        USER_ID = result.userId || '';
        console.log(
          `🔧 Settings loaded: API=${API_BASE}, User=${USER_ID ? 'configured' : 'not set'}`
        );
      } catch (error) {
        console.error('Failed to load settings:', error);
      }
    }

    // Listen for storage changes (when user connects via popup)
    browserAPI.storage.onChanged.addListener((changes, areaName) => {
      if (areaName === 'local') {
        if (changes.userId) {
          USER_ID = changes.userId.newValue || '';
          console.log(`🔄 User ID updated: ${USER_ID ? 'configured' : 'not set'}`);
        }
        if (changes.apiEndpoint) {
          API_BASE = changes.apiEndpoint.newValue || 'https://api.safehaiven.com';
          console.log(`🔄 API endpoint updated: ${API_BASE}`);
        }
      }
    });

    // Haiven UI Injection - Inline button near send button + fallback floating button
    function injectHaivenButton() {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea || document.getElementById('haiven-inline-btn')) return;

      // Try to inject inline button near the input area
      const inlineButton = createInlineButton();
      const injected = injectInlineButton(inlineButton);

      // Always create floating button as backup/alternative access
      if (!document.getElementById('haiven-floating-btn')) {
        createFloatingButton();
      }

      if (injected) {
        console.log('🧠 Haiven: Inline button injected successfully');
      }
    }

    // Create the small inline button (appears near send button)
    function createInlineButton() {
      const button = document.createElement('button');
      button.id = 'haiven-inline-btn';
      button.title = 'Haiven (Ctrl+H)';
      button.innerHTML = '🧠';
      button.style.cssText = `
        background: linear-gradient(135deg, #f59e0b, #d97706);
        border: none;
        border-radius: 8px;
        width: 32px;
        height: 32px;
        font-size: 16px;
        cursor: pointer;
        display: flex;
        align-items: center;
        justify-content: center;
        box-shadow: 0 2px 8px rgba(245, 158, 11, 0.3);
        transition: all 0.2s;
        flex-shrink: 0;
        margin: 0 4px;
      `;

      button.addEventListener('mouseenter', () => {
        button.style.transform = 'scale(1.1)';
        button.style.boxShadow = '0 4px 12px rgba(245, 158, 11, 0.5)';
      });

      button.addEventListener('mouseleave', () => {
        button.style.transform = 'scale(1)';
        button.style.boxShadow = '0 2px 8px rgba(245, 158, 11, 0.3)';
      });

      button.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        showHaivenPanel();
      });

      return button;
    }

    // Try to inject inline button near the composer/send area
    function injectInlineButton(button) {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) {
        console.log('🧠 Haiven: No input area found for inline button');
        return false;
      }

      // Strategy: Find the parent container of the input and inject there
      // Different platforms have different structures

      // ChatGPT: Find the form or composer container
      if (currentPlatform.name === 'ChatGPT') {
        // Try multiple strategies for ChatGPT's changing DOM
        const form = inputArea.closest('form');
        if (form) {
          // Strategy 1: Find send button and inject next to it
          const sendButton =
            form.querySelector('button[data-testid="send-button"]') ||
            form.querySelector('button[aria-label*="Send"]') ||
            form.querySelector('button[type="submit"]');

          if (sendButton && sendButton.parentElement) {
            sendButton.parentElement.insertBefore(button, sendButton);
            console.log('🧠 Haiven: Injected next to send button');
            return true;
          }

          // Strategy 2: Find any button group in the form
          const buttonGroup = form.querySelector('[class*="flex"][class*="gap"]');
          if (buttonGroup) {
            buttonGroup.insertBefore(button, buttonGroup.firstChild);
            console.log('🧠 Haiven: Injected into button group');
            return true;
          }

          // Strategy 3: Absolute position fallback
          const wrapper = document.createElement('div');
          wrapper.style.cssText = 'position: absolute; bottom: 12px; left: 12px; z-index: 10;';
          wrapper.appendChild(button);
          form.style.position = 'relative';
          form.appendChild(wrapper);
          console.log('🧠 Haiven: Injected with absolute positioning');
          return true;
        }
      }

      // Claude: Find the editor container
      if (currentPlatform.name === 'Claude') {
        const editorContainer =
          inputArea.closest('[class*="editor"]') ||
          inputArea.closest('[class*="input"]') ||
          inputArea.parentElement;
        if (editorContainer) {
          const wrapper = document.createElement('div');
          wrapper.style.cssText = 'position: absolute; bottom: 8px; left: 8px; z-index: 10;';
          wrapper.appendChild(button);
          editorContainer.style.position = 'relative';
          editorContainer.appendChild(wrapper);
          return true;
        }
      }

      // Grok: Find the composer
      if (currentPlatform.name === 'Grok') {
        const composer =
          document.querySelector(currentPlatform.composerSelector) ||
          inputArea.closest('[class*="composer"]') ||
          inputArea.parentElement;
        if (composer) {
          const wrapper = document.createElement('div');
          wrapper.style.cssText = 'position: absolute; bottom: 8px; left: 8px; z-index: 10;';
          wrapper.appendChild(button);
          composer.style.position = 'relative';
          composer.appendChild(wrapper);
          return true;
        }
      }

      // Generic fallback: position near input
      const wrapper = document.createElement('div');
      wrapper.id = 'haiven-inline-wrapper';
      wrapper.style.cssText = 'position: absolute; bottom: 8px; left: 8px; z-index: 999999;';
      wrapper.appendChild(button);

      const parent = inputArea.parentElement;
      if (parent) {
        parent.style.position = 'relative';
        parent.appendChild(wrapper);
        return true;
      }

      return false;
    }

    // Create floating button as backup
    function createFloatingButton() {
      const button = document.createElement('button');
      button.id = 'haiven-floating-btn';
      button.textContent = '🧠';
      button.title = 'Haiven Panel (Ctrl+H)';
      button.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        z-index: 999999;
        background: linear-gradient(135deg, #f59e0b, #d97706);
        color: white;
        border: none;
        border-radius: 50%;
        width: 48px;
        height: 48px;
        font-size: 22px;
        cursor: pointer;
        box-shadow: 0 4px 20px rgba(245, 158, 11, 0.4);
        transition: all 0.2s;
        display: flex;
        align-items: center;
        justify-content: center;
      `;

      button.addEventListener('mouseenter', () => {
        button.style.transform = 'scale(1.1)';
        button.style.boxShadow = '0 6px 24px rgba(245, 158, 11, 0.5)';
      });

      button.addEventListener('mouseleave', () => {
        button.style.transform = 'scale(1)';
        button.style.boxShadow = '0 4px 20px rgba(245, 158, 11, 0.4)';
      });

      button.addEventListener('click', () => showHaivenPanel());

      document.body.appendChild(button);
    }

    // Haiven Control Panel - Premium 10/10 Redesign
    function showHaivenPanel() {
      if (document.getElementById('haiven-panel')) {
        document.getElementById('haiven-panel').remove();
        return;
      }

      // Premium CSS variables - Refined for luxury feel
      const cssVars = {
        // Brand colors
        amber: '#f59e0b',
        amberDark: '#d97706',
        amberLight: '#fbbf24',
        gradient: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
        gradientSubtle: 'linear-gradient(135deg, rgba(245, 158, 11, 0.15) 0%, rgba(217, 119, 6, 0.15) 100%)',
        // Glows
        glow: 'rgba(245, 158, 11, 0.5)',
        glowSubtle: 'rgba(245, 158, 11, 0.15)',
        glowStrong: 'rgba(245, 158, 11, 0.7)',
        // Backgrounds - deeper, richer
        bgPrimary: 'rgba(13, 13, 18, 0.97)',
        bgSecondary: 'rgba(255, 255, 255, 0.025)',
        bgTertiary: 'rgba(255, 255, 255, 0.05)',
        bgHover: 'rgba(245, 158, 11, 0.1)',
        bgCard: 'rgba(255, 255, 255, 0.02)',
        // Text - refined hierarchy
        textPrimary: 'rgba(255, 255, 255, 0.97)',
        textSecondary: 'rgba(255, 255, 255, 0.65)',
        textTertiary: 'rgba(255, 255, 255, 0.4)',
        textMuted: 'rgba(255, 255, 255, 0.25)',
        // Borders
        borderSubtle: 'rgba(255, 255, 255, 0.06)',
        borderAccent: 'rgba(245, 158, 11, 0.25)',
        borderGlow: 'rgba(245, 158, 11, 0.4)',
        // Shadows - layered for depth
        shadowSm: '0 1px 2px rgba(0,0,0,0.2)',
        shadowMd: '0 4px 12px rgba(0,0,0,0.3)',
        shadowLg: '0 12px 40px rgba(0,0,0,0.4)',
        shadowXl: '0 25px 80px -12px rgba(0,0,0,0.6)',
        shadowGlow: '0 0 40px rgba(245, 158, 11, 0.15)',
        // Animations
        spring: 'cubic-bezier(0.34, 1.56, 0.64, 1)',
        smooth: 'cubic-bezier(0.4, 0, 0.2, 1)',
        bounce: 'cubic-bezier(0.68, -0.55, 0.265, 1.55)'
      };

      const panel = document.createElement('div');
      panel.id = 'haiven-panel';
      panel.style.cssText = `
        position: fixed;
        bottom: 80px;
        right: 20px;
        width: 360px;
        background: ${cssVars.bgPrimary};
        backdrop-filter: blur(60px) saturate(200%);
        -webkit-backdrop-filter: blur(60px) saturate(200%);
        border-radius: 24px;
        border: 1px solid ${cssVars.borderAccent};
        box-shadow:
          0 0 0 1px rgba(255,255,255,0.03) inset,
          ${cssVars.shadowXl},
          ${cssVars.shadowGlow};
        z-index: 999998;
        overflow: hidden;
        font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'Segoe UI', Roboto, sans-serif;
        color: ${cssVars.textPrimary};
        animation: haivenPanelEntrance 0.6s ${cssVars.spring} forwards;
      `;

      // Ambient glow overlay - warm amber tones
      const glowOverlay = createElement('div', {
        style: {
          position: 'absolute',
          inset: '0',
          opacity: '1',
          pointerEvents: 'none',
          zIndex: '0',
          background: `
            radial-gradient(ellipse 80% 50% at 50% -20%, rgba(245, 158, 11, 0.12) 0%, transparent 50%),
            radial-gradient(ellipse 60% 40% at 80% 100%, rgba(217, 119, 6, 0.08) 0%, transparent 50%)
          `,
          animation: 'haivenGlow 4s ease-in-out infinite'
        }
      });
      panel.appendChild(glowOverlay);

      // Subtle noise texture for premium feel
      const noiseOverlay = createElement('div', {
        style: {
          position: 'absolute',
          inset: '0',
          opacity: '0.015',
          pointerEvents: 'none',
          zIndex: '0',
          backgroundImage: `url("data:image/svg+xml,%3Csvg viewBox='0 0 256 256' xmlns='http://www.w3.org/2000/svg'%3E%3Cfilter id='noise'%3E%3CfeTurbulence type='fractalNoise' baseFrequency='0.9' numOctaves='4' stitchTiles='stitch'/%3E%3C/filter%3E%3Crect width='100%25' height='100%25' filter='url(%23noise)'/%3E%3C/svg%3E")`,
          mixBlendMode: 'overlay'
        }
      });
      panel.appendChild(noiseOverlay);

      // SECURITY: Build panel with safe DOM methods
      // Create premium header with refined design
      const header = createElement(
        'div',
        {
          style: {
            position: 'relative',
            zIndex: '1',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'space-between',
            padding: '20px 24px',
            borderBottom: `1px solid ${cssVars.borderSubtle}`,
            background: 'linear-gradient(180deg, rgba(255,255,255,0.02) 0%, transparent 100%)'
          }
        },
        [
          createElement(
            'div',
            {
              style: { display: 'flex', alignItems: 'center', gap: '14px' }
            },
            [
              // Premium logo container with glow
              createElement('div', {
                style: {
                  width: '42px',
                  height: '42px',
                  borderRadius: '12px',
                  background: cssVars.gradientSubtle,
                  display: 'flex',
                  alignItems: 'center',
                  justifyContent: 'center',
                  border: `1px solid ${cssVars.borderAccent}`,
                  boxShadow: `0 0 20px ${cssVars.glowSubtle}`
                }
              }, [
                createElement('span', {
                  textContent: '🧠',
                  style: {
                    fontSize: '22px',
                    animation: 'haivenIconPulse 3s ease-in-out infinite'
                  }
                })
              ]),
              createElement('div', {}, [
                createElement('div', {
                  textContent: 'Haiven',
                  style: {
                    fontSize: '19px',
                    fontWeight: '700',
                    color: cssVars.textPrimary,
                    letterSpacing: '-0.5px',
                    lineHeight: '1.2'
                  }
                }),
                // Status pill
                createElement(
                  'div',
                  {
                    style: {
                      display: 'inline-flex',
                      alignItems: 'center',
                      gap: '6px',
                      fontSize: '11px',
                      color: cssVars.textTertiary,
                      marginTop: '4px',
                      background: 'rgba(16, 185, 129, 0.1)',
                      padding: '3px 10px 3px 8px',
                      borderRadius: '20px',
                      border: '1px solid rgba(16, 185, 129, 0.2)'
                    }
                  },
                  [
                    // Pulsing status dot with ring
                    createElement('div', {
                      style: {
                        position: 'relative',
                        width: '8px',
                        height: '8px'
                      }
                    }, [
                      createElement('span', {
                        style: {
                          position: 'absolute',
                          inset: '0',
                          background: '#10B981',
                          borderRadius: '50%'
                        }
                      }),
                      createElement('span', {
                        style: {
                          position: 'absolute',
                          inset: '-2px',
                          border: '2px solid #10B981',
                          borderRadius: '50%',
                          animation: 'haivenRingPulse 2s ease-out infinite'
                        }
                      })
                    ]),
                    createElement('span', {
                      textContent: currentPlatform.name,
                      style: { fontWeight: '500' }
                    })
                  ]
                )
              ])
            ]
          ),
          // Close button - refined
          createElement('button', {
            id: 'haiven-close',
            textContent: '×',
            style: {
              width: '32px',
              height: '32px',
              display: 'flex',
              alignItems: 'center',
              justifyContent: 'center',
              background: cssVars.bgSecondary,
              border: `1px solid ${cssVars.borderSubtle}`,
              color: cssVars.textTertiary,
              cursor: 'pointer',
              borderRadius: '10px',
              transition: `all 0.2s ${cssVars.smooth}`,
              fontSize: '20px',
              lineHeight: '1'
            }
          })
        ]
      );

      // Create primary CTA with premium shimmer effect
      const ctaButton = createElement(
        'button',
        {
          id: 'haiven-inject-context',
          style: {
            width: '100%',
            padding: '18px 24px',
            background: cssVars.gradient,
            border: 'none',
            borderRadius: '16px',
            color: 'white',
            fontSize: '15px',
            fontWeight: '600',
            letterSpacing: '-0.2px',
            cursor: 'pointer',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            gap: '10px',
            position: 'relative',
            overflow: 'hidden',
            boxShadow: `
              0 2px 4px rgba(0,0,0,0.1),
              0 4px 12px rgba(245, 158, 11, 0.25),
              0 0 0 1px rgba(255,255,255,0.1) inset
            `,
            transition: `all 0.3s ${cssVars.spring}`
          }
        },
        [
          createElement('span', {
            textContent: '✨',
            style: { fontSize: '16px' }
          }),
          createElement('span', {
            id: 'haiven-cta-text',
            textContent: 'Inject Smart Context'
          }),
          // Shimmer effect overlay
          createElement('div', {
            style: {
              position: 'absolute',
              top: '0',
              left: '0',
              right: '0',
              bottom: '0',
              background: 'linear-gradient(90deg, transparent 0%, rgba(255,255,255,0.2) 50%, transparent 100%)',
              transform: 'translateX(-100%)',
              animation: 'haivenShimmer 3s ease-in-out infinite',
              pointerEvents: 'none'
            }
          })
        ]
      );

      const ctaContainer = createElement('div', { style: { marginBottom: '16px' } }, [
        ctaButton,
        createElement('div', {
          id: 'haiven-context-status',
          style: {
            marginTop: '10px',
            fontSize: '12px',
            color: cssVars.textTertiary,
            textAlign: 'center',
            opacity: '0.8'
          }
        })
      ]);

      // Keyboard shortcut helper - refined
      const keyboardHelper = createElement(
        'div',
        {
          style: {
            fontSize: '11px',
            color: cssVars.textMuted,
            textAlign: 'center',
            marginTop: '14px',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            gap: '4px'
          }
        },
        [
          createElement('span', { textContent: 'or press' }),
          createElement('kbd', {
            textContent: '⌘I',
            style: {
              background: 'rgba(255,255,255,0.04)',
              padding: '3px 8px',
              borderRadius: '6px',
              fontFamily: "'SF Mono', 'Menlo', monospace",
              fontSize: '10px',
              fontWeight: '500',
              border: `1px solid ${cssVars.borderSubtle}`,
              color: cssVars.textTertiary,
              marginLeft: '4px'
            }
          })
        ]
      );

      // Elegant divider with fade
      const divider = createElement('div', {
        style: {
          height: '1px',
          background: `linear-gradient(90deg, transparent 0%, ${cssVars.borderSubtle} 20%, ${cssVars.borderSubtle} 80%, transparent 100%)`,
          margin: '20px 0'
        }
      });

      // Create premium secondary action button (glass-morphic style)
      const createSecondaryButton = (id, icon, label) => {
        const btn = createElement(
          'button',
          {
            id,
            style: {
              display: 'flex',
              flexDirection: 'column',
              alignItems: 'center',
              justifyContent: 'center',
              gap: '8px',
              padding: '18px 12px',
              background: cssVars.bgCard,
              border: `1px solid ${cssVars.borderSubtle}`,
              borderRadius: '14px',
              color: cssVars.textSecondary,
              fontSize: '12px',
              fontWeight: '500',
              letterSpacing: '-0.1px',
              cursor: 'pointer',
              transition: `all 0.25s ${cssVars.smooth}`,
              position: 'relative',
              overflow: 'hidden'
            }
          },
          [
            createElement('span', {
              textContent: icon,
              style: {
                fontSize: '22px',
                transition: `transform 0.3s ${cssVars.spring}`
              }
            }),
            createElement('span', { textContent: label })
          ]
        );

        // Premium hover effects
        btn.addEventListener('mouseenter', () => {
          btn.style.background = cssVars.bgHover;
          btn.style.borderColor = cssVars.borderAccent;
          btn.style.color = cssVars.textPrimary;
          btn.style.transform = 'translateY(-3px)';
          btn.style.boxShadow = `0 8px 24px rgba(245, 158, 11, 0.12), 0 0 0 1px ${cssVars.borderAccent}`;
          btn.querySelector('span').style.transform = 'scale(1.1)';
        });
        btn.addEventListener('mouseleave', () => {
          btn.style.background = cssVars.bgCard;
          btn.style.borderColor = cssVars.borderSubtle;
          btn.style.color = cssVars.textSecondary;
          btn.style.transform = 'translateY(0)';
          btn.style.boxShadow = 'none';
          btn.querySelector('span').style.transform = 'scale(1)';
        });

        return btn;
      };

      // 2x2 Actions grid - refined spacing
      const actionsGrid = createElement(
        'div',
        {
          style: {
            display: 'grid',
            gridTemplateColumns: '1fr 1fr',
            gap: '10px'
          }
        },
        [
          createSecondaryButton('haiven-optimize-prompt', '🎯', 'Optimize'),
          createSecondaryButton('haiven-capture-conversation', '💾', 'Save'),
          createSecondaryButton('haiven-recall-search', '🔍', 'Search'),
          createSecondaryButton('haiven-continue-llm', '🔄', 'Continue')
        ]
      );

      // Assemble content area
      const content = createElement(
        'div',
        {
          style: {
            position: 'relative',
            zIndex: '1',
            padding: '20px 24px 24px'
          }
        },
        [ctaContainer, keyboardHelper, divider, actionsGrid]
      );

      // Premium footer with refined toggle
      const footer = createElement(
        'div',
        {
          style: {
            position: 'relative',
            zIndex: '1',
            padding: '14px 24px',
            borderTop: `1px solid ${cssVars.borderSubtle}`,
            background: 'rgba(0,0,0,0.15)',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'space-between'
          }
        },
        [
          createElement('div', {
            style: { display: 'flex', alignItems: 'center', gap: '8px' }
          }, [
            createElement('span', {
              textContent: '⚡',
              style: { fontSize: '12px', opacity: '0.6' }
            }),
            createElement('span', {
              textContent: 'Auto-inject context',
              style: {
                fontSize: '12px',
                color: cssVars.textSecondary,
                fontWeight: '500'
              }
            })
          ]),
          // Premium toggle switch
          createElement(
            'div',
            {
              id: 'haiven-toggle-container',
              style: {
                position: 'relative',
                width: '48px',
                height: '26px',
                background: cssVars.bgTertiary,
                borderRadius: '13px',
                cursor: 'pointer',
                transition: `all 0.3s ${cssVars.smooth}`,
                border: `1px solid ${cssVars.borderSubtle}`
              }
            },
            [
              createElement('input', {
                type: 'checkbox',
                id: 'haiven-auto-inject',
                style: { opacity: '0', width: '0', height: '0', position: 'absolute' }
              }),
              createElement('span', {
                id: 'haiven-toggle-knob',
                style: {
                  position: 'absolute',
                  top: '3px',
                  left: '3px',
                  width: '18px',
                  height: '18px',
                  background: 'white',
                  borderRadius: '50%',
                  transition: `all 0.3s ${cssVars.spring}`,
                  boxShadow: '0 2px 6px rgba(0,0,0,0.25)'
                }
              })
            ]
          )
        ]
      );

      panel.appendChild(header);
      panel.appendChild(content);
      panel.appendChild(footer);

      document.body.appendChild(panel);

      // Event handlers - refined interactions
      const closeBtn = document.getElementById('haiven-close');
      closeBtn.addEventListener('mouseenter', () => {
        closeBtn.style.background = 'rgba(255,255,255,0.08)';
        closeBtn.style.color = cssVars.textPrimary;
        closeBtn.style.borderColor = cssVars.borderAccent;
        closeBtn.style.transform = 'scale(1.05)';
      });
      closeBtn.addEventListener('mouseleave', () => {
        closeBtn.style.background = cssVars.bgSecondary;
        closeBtn.style.color = cssVars.textTertiary;
        closeBtn.style.borderColor = cssVars.borderSubtle;
        closeBtn.style.transform = 'scale(1)';
      });
      closeBtn.addEventListener('click', () => panel.remove());

      // Primary CTA hover effects - premium feel
      const ctaBtn = document.getElementById('haiven-inject-context');
      ctaBtn.addEventListener('mouseenter', () => {
        ctaBtn.style.transform = 'translateY(-2px) scale(1.01)';
        ctaBtn.style.boxShadow = `
          0 4px 8px rgba(0,0,0,0.15),
          0 8px 24px rgba(245, 158, 11, 0.35),
          0 0 0 1px rgba(255,255,255,0.15) inset
        `;
      });
      ctaBtn.addEventListener('mouseleave', () => {
        ctaBtn.style.transform = 'translateY(0) scale(1)';
        ctaBtn.style.boxShadow = `
          0 2px 4px rgba(0,0,0,0.1),
          0 4px 12px rgba(245, 158, 11, 0.25),
          0 0 0 1px rgba(255,255,255,0.1) inset
        `;
      });
      ctaBtn.addEventListener('click', injectSmartContext);

      document.getElementById('haiven-optimize-prompt').addEventListener('click', optimizePrompt);
      document
        .getElementById('haiven-capture-conversation')
        .addEventListener('click', captureConversation);
      document.getElementById('haiven-recall-search').addEventListener('click', showRecallModal);
      document
        .getElementById('haiven-continue-llm')
        .addEventListener('click', showContinueConversationModal);

      // Premium toggle functionality with glow effect
      const toggleContainer = document.getElementById('haiven-toggle-container');
      const toggleInput = document.getElementById('haiven-auto-inject');
      const toggleKnob = document.getElementById('haiven-toggle-knob');

      const updateToggleVisual = (isActive) => {
        if (isActive) {
          toggleContainer.style.background = cssVars.gradient;
          toggleContainer.style.borderColor = 'transparent';
          toggleContainer.style.boxShadow = `0 0 16px ${cssVars.glowSubtle}`;
          toggleKnob.style.transform = 'translateX(22px)';
          toggleKnob.style.boxShadow = '0 2px 8px rgba(0,0,0,0.3)';
        } else {
          toggleContainer.style.background = cssVars.bgTertiary;
          toggleContainer.style.borderColor = cssVars.borderSubtle;
          toggleContainer.style.boxShadow = 'none';
          toggleKnob.style.transform = 'translateX(0)';
          toggleKnob.style.boxShadow = '0 2px 6px rgba(0,0,0,0.25)';
        }
      };

      // Load auto-inject preference
      browserAPI.storage.local.get(['autoInject'], (result) => {
        const isActive = result.autoInject || false;
        toggleInput.checked = isActive;
        updateToggleVisual(isActive);
      });

      toggleContainer.addEventListener('click', () => {
        toggleInput.checked = !toggleInput.checked;
        updateToggleVisual(toggleInput.checked);
        browserAPI.storage.local.set({ autoInject: toggleInput.checked });
      });
    }

    // Smart Context Injection
    async function injectSmartContext() {
      const statusEl = document.getElementById('haiven-context-status');
      const inputArea = document.querySelector(currentPlatform.inputSelector);

      if (!inputArea) {
        showNotification('❌ Could not find input area', 'error');
        return;
      }

      // Ensure we have the latest userId from storage
      let userId = USER_ID;
      if (!userId) {
        const result = await browserAPI.storage.local.get(['userId']);
        userId = result.userId || '';
        if (userId) {
          USER_ID = userId; // Update cached value
          console.log('🔄 Retrieved userId from storage for context injection');
        }
      }

      if (!userId) {
        showNotification('⚠️ Please connect your account first', 'warning');
        statusEl.textContent = '❌ Not connected';
        return;
      }

      try {
        statusEl.textContent = '⏳ Retrieving context...';

        // Get current input to understand query intent
        const currentInput = inputArea.value || inputArea.textContent || '';

        const response = await safeFetch(
          `${API_BASE}/api/memory/smart-context`,
          {
            method: 'POST',
            headers: await getAuthHeaders(),
            body: JSON.stringify({
              userId: userId,
              query: currentInput || 'What do you know about me?',
              platform: currentPlatform.name, // Send platform for optimized formatting
              maxTokens: Math.floor(currentPlatform.maxTokens * 0.3) // Use 30% of token budget
            })
          },
          8000
        ); // 8 second timeout for context injection

        const data = await response.json();

        if (!data.success) {
          throw new Error(data.error || 'Failed to get context');
        }

        // Translate context to platform-specific format
        const translatedContext = translateContextForPlatform(data.context, currentPlatform);

        // Inject into input
        if (inputArea.tagName === 'TEXTAREA' || inputArea.tagName === 'INPUT') {
          inputArea.value = `${translatedContext}\n\n${currentInput}`;
        } else {
          inputArea.textContent = `${translatedContext}\n\n${currentInput}`;
        }

        // Trigger input event
        inputArea.dispatchEvent(new Event('input', { bubbles: true }));

        statusEl.textContent = `✅ Injected ${data.memoriesUsed} memories (${data.totalTokens} tokens)`;
        showNotification(`✨ Context injected: ${data.memoriesUsed} memories`, 'success');
      } catch (error) {
        console.error('Context injection error:', error);
        statusEl.textContent = '❌ Failed to inject context';
        showNotification('❌ Failed to inject context', 'error');
      }
    }

    // Translate context to platform-specific format
    function translateContextForPlatform(context, platform) {
      switch (platform.promptStyle) {
        case 'xml-structured': // Claude
          return `<context>\n${context}\n</context>\n\nPlease use the above context when responding:`;

        case 'conversational': // ChatGPT, Gemini
          return `Here's relevant context about me:\n\n${context}\n\nConsidering this context:`;

        case 'direct': // Grok
          return `Context: ${context}\n\nQuestion:`;

        case 'research-focused': // Perplexity
          return `Background information:\n${context}\n\nSearch query:`;

        default:
          return context;
      }
    }

    // Optimize Prompt
    async function optimizePrompt() {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) return;

      const currentPrompt = inputArea.value || inputArea.textContent || '';
      if (!currentPrompt.trim()) {
        showNotification('⚠️ Enter a prompt first', 'warning');
        return;
      }

      try {
        showNotification('🎯 Optimizing prompt...', 'info');

        const response = await fetch(`${API_BASE}/api/prompts/optimize`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            prompt: currentPrompt,
            platform: currentPlatform.name,
            promptStyle: currentPlatform.promptStyle
          })
        });

        const data = await response.json();

        if (data.success) {
          // Show optimization suggestions
          showOptimizationSuggestions(data.optimizedPrompt, data.suggestions, currentPrompt);
        }
      } catch (error) {
        console.error('Optimization error:', error);
        showNotification('❌ Optimization failed', 'error');
      }
    }

    // Show optimization suggestions
    function showOptimizationSuggestions(optimized, suggestions, original) {
      const modal = document.createElement('div');
      modal.style.cssText = `
      position: fixed;
      top: 50%;
      left: 50%;
      transform: translate(-50%, -50%);
      width: 600px;
      max-height: 80vh;
      background: #18181B;
      border: 1px solid rgba(139, 92, 246, 0.3);
      border-radius: 16px;
      box-shadow: 0 8px 40px rgba(0, 0, 0, 0.7);
      z-index: 999999;
      overflow-y: auto;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      color: white;
      padding: 24px;
    `;

      modal.innerHTML = `
      <div style="margin-bottom: 20px;">
        <h2 style="margin: 0 0 8px 0; font-size: 20px; font-weight: 600;">🎯 Prompt Optimization</h2>
        <p style="margin: 0; font-size: 14px; color: rgba(255,255,255,0.6);">
          Optimized for ${currentPlatform.name}
        </p>
      </div>

      <div style="margin-bottom: 20px;">
        <h3 style="font-size: 14px; font-weight: 600; margin-bottom: 8px;">Original:</h3>
        <div style="padding: 12px; background: rgba(255,255,255,0.05); border-radius: 8px; font-size: 13px; line-height: 1.6;">
          ${escapeHtml(original)}
        </div>
      </div>

      <div style="margin-bottom: 20px;">
        <h3 style="font-size: 14px; font-weight: 600; margin-bottom: 8px;">Optimized:</h3>
        <div style="padding: 12px; background: rgba(139, 92, 246, 0.1); border: 1px solid rgba(139, 92, 246, 0.3); border-radius: 8px; font-size: 13px; line-height: 1.6;">
          ${escapeHtml(optimized)}
        </div>
      </div>

      <div style="margin-bottom: 24px;">
        <h3 style="font-size: 14px; font-weight: 600; margin-bottom: 8px;">💡 Improvements:</h3>
        ${suggestions
          .map(
            (s) => `
          <div style="padding: 8px 12px; margin-bottom: 6px; background: rgba(16, 185, 129, 0.1); border-left: 3px solid #10B981; border-radius: 4px; font-size: 13px;">
            ${escapeHtml(s)}
          </div>
        `
          )
          .join('')}
      </div>

      <div style="display: flex; gap: 12px;">
        <button id="apply-optimized" style="flex: 1; background: linear-gradient(135deg, #f59e0b, #d97706); color: white; border: none; border-radius: 8px; padding: 12px; font-size: 14px; font-weight: 600; cursor: pointer;">
          Apply Optimized
        </button>
        <button id="close-modal" style="flex: 1; background: rgba(255,255,255,0.1); color: white; border: none; border-radius: 8px; padding: 12px; font-size: 14px; font-weight: 600; cursor: pointer;">
          Cancel
        </button>
      </div>
    `;

      document.body.appendChild(modal);

      document.getElementById('apply-optimized').addEventListener('click', () => {
        const inputArea = document.querySelector(currentPlatform.inputSelector);
        if (inputArea.tagName === 'TEXTAREA' || inputArea.tagName === 'INPUT') {
          inputArea.value = optimized;
        } else {
          inputArea.textContent = optimized;
        }
        inputArea.dispatchEvent(new Event('input', { bubbles: true }));
        modal.remove();
        showNotification('✅ Prompt optimized!', 'success');
      });

      document.getElementById('close-modal').addEventListener('click', () => modal.remove());
    }

    // Capture Conversation
    async function captureConversation() {
      try {
        showNotification('💾 Capturing conversation...', 'info');

        const messages = Array.from(document.querySelectorAll(currentPlatform.messageSelector))
          .map((el) => el.textContent.trim())
          .filter((text) => text.length > 0);

        if (messages.length === 0) {
          showNotification('⚠️ No conversation to capture', 'warning');
          return;
        }

        const conversationContext = messages.join('\n\n---\n\n');

        const response = await safeFetch(`${API_BASE}/api/memory/store`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            content: conversationContext,
            source: `${currentPlatform.name.toLowerCase()}-conversation`,
            metadata: {
              platform: currentPlatform.name,
              messageCount: messages.length,
              capturedAt: new Date().toISOString(),
              url: window.location.href
            }
          })
        });

        const data = await response.json();

        if (data.success) {
          // Track LLM usage when capturing conversation
          await trackLLMUsage(currentPlatform.name, messages);
          showNotification(`✅ Captured ${messages.length} messages`, 'success');
        }
      } catch (error) {
        console.error('Capture error:', error);
        showNotification('❌ Failed to capture conversation', 'error');
      }
    }

    // ========================================
    // LLM USAGE TRACKING
    // Track what users use each LLM for
    // ========================================

    // Track last usage to avoid duplicate tracking
    let lastTrackedTaskTypes = [];
    let usageTrackingCooldown = false;

    // Detect task types from text content
    function detectTaskTypes(text) {
      const lowerText = text.toLowerCase();
      const taskTypes = [];

      if (
        /\b(code|function|debug|implement|error|fix|class|method|api|syntax|variable|loop|array|object|import|export|async|await|promise|typescript|javascript|python|react|node|npm|git)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('coding');
      }
      if (
        /\b(design|ui|ux|layout|color|font|style|visual|mockup|wireframe|prototype|figma|css|responsive|animation)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('design');
      }
      if (
        /\b(video|image|picture|photo|generate|create image|dalle|midjourney|stable diffusion|render|3d|animation|veo)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('media');
      }
      if (
        /\b(research|study|paper|citation|source|article|journal|academic|reference|bibliography|fact|evidence)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('research');
      }
      if (
        /\b(write|essay|article|blog|content|story|narrative|draft|copy|email|message|document|report)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('writing');
      }
      if (
        /\b(brainstorm|idea|think|plan|strategy|concept|creative|innovation|solution|approach|options)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('brainstorming');
      }
      if (
        /\b(math|calculate|equation|formula|statistics|data|analyze|chart|graph|excel|spreadsheet)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('analysis');
      }
      if (
        /\b(learn|explain|teach|understand|tutorial|how to|what is|why does|example|concept)\b/.test(
          lowerText
        )
      ) {
        taskTypes.push('learning');
      }

      return taskTypes.length > 0 ? taskTypes : ['general'];
    }

    // Track usage from real-time input (called periodically as user types)
    async function trackRealtimeUsage(inputText) {
      // Cooldown to avoid excessive tracking (track max once per 30 seconds)
      if (usageTrackingCooldown || !USER_ID || inputText.length < 30) return;

      const taskTypes = detectTaskTypes(inputText);

      // Only track if task types changed
      if (JSON.stringify(taskTypes) === JSON.stringify(lastTrackedTaskTypes)) return;

      lastTrackedTaskTypes = taskTypes;
      usageTrackingCooldown = true;

      try {
        await fetch(`${API_BASE}/api/llm/track-usage`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            platform: currentPlatform.name,
            taskTypes: taskTypes,
            messageCount: 1,
            source: 'realtime',
            timestamp: new Date().toISOString()
          })
        });

        console.log(
          `📊 Real-time usage tracked: ${currentPlatform.name} for ${taskTypes.join(', ')}`
        );
      } catch (error) {
        console.error('Real-time usage tracking error:', error);
      }

      // Reset cooldown after 30 seconds
      setTimeout(() => {
        usageTrackingCooldown = false;
      }, 30000);
    }

    async function trackLLMUsage(platform, messages) {
      try {
        // Analyze conversation topics to learn user preferences
        const conversationText = messages.join(' ').toLowerCase();
        const taskTypes = detectTaskTypes(conversationText);

        // Store usage pattern
        await fetch(`${API_BASE}/api/llm/track-usage`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            platform: platform,
            taskTypes: taskTypes,
            messageCount: messages.length,
            source: 'capture',
            timestamp: new Date().toISOString()
          })
        });

        console.log(`📊 LLM Usage tracked: ${platform} for ${taskTypes.join(', ')}`);
      } catch (error) {
        console.error('LLM usage tracking error:', error);
      }
    }

    // ========================================
    // CONTINUE CONVERSATION IN ANOTHER LLM
    // Export conversation for seamless transition
    // ========================================

    async function showContinueConversationModal() {
      // Get current conversation
      const userMessages = Array.from(document.querySelectorAll(currentPlatform.messageSelector))
        .map((el) => el.textContent.trim())
        .filter((text) => text.length > 0);

      const assistantMessages = Array.from(
        document.querySelectorAll(currentPlatform.assistantSelector)
      )
        .map((el) => el.textContent.trim())
        .filter((text) => text.length > 0);

      if (userMessages.length === 0) {
        showNotification('⚠️ No conversation to continue', 'warning');
        return;
      }

      // Fetch user's LLM preferences
      let llmPreferences = {};
      try {
        const response = await fetch(`${API_BASE}/api/llm/preferences?userId=${USER_ID}`);
        const data = await response.json();
        llmPreferences = data.preferences || {};
      } catch (error) {
        console.error('Failed to fetch LLM preferences:', error);
      }

      // LLM profiles with strengths
      const LLM_PROFILES = {
        ChatGPT: {
          strengths: ['General conversation', 'Creative writing', 'Brainstorming', 'Web search'],
          icon: '🤖',
          url: 'https://chat.openai.com'
        },
        Claude: {
          strengths: ['Coding', 'Long documents', 'Analysis', 'Technical writing'],
          icon: '🧠',
          url: 'https://claude.ai'
        },
        Grok: {
          strengths: ['Real-time info', 'News', 'Current events', 'Twitter context'],
          icon: '⚡',
          url: 'https://x.com/i/grok'
        },
        Perplexity: {
          strengths: ['Research', 'Citations', 'Factual queries', 'Academic'],
          icon: '🔍',
          url: 'https://perplexity.ai'
        },
        Gemini: {
          strengths: ['Multimodal', 'Images', 'Video', 'Large context'],
          icon: '💫',
          url: 'https://gemini.google.com'
        }
      };

      // Remove current platform from options
      const otherLLMs = Object.keys(LLM_PROFILES).filter((llm) => llm !== currentPlatform.name);

      const modal = document.createElement('div');
      modal.id = 'haiven-continue-modal';
      modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        backdrop-filter: blur(8px);
        z-index: 9999999;
        display: flex;
        align-items: center;
        justify-content: center;
        animation: fadeIn 0.3s ease-out;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      `;

      modal.innerHTML = `
        <div style="
          background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
          border-radius: 20px;
          padding: 32px;
          max-width: 600px;
          width: 90%;
          max-height: 85vh;
          overflow-y: auto;
          box-shadow: 0 20px 60px rgba(245, 158, 11, 0.3);
          border: 2px solid rgba(245, 158, 11, 0.2);
          color: white;
        ">
          <div style="text-align: center; margin-bottom: 24px;">
            <div style="font-size: 40px; margin-bottom: 12px;">🔄</div>
            <h2 style="font-size: 22px; margin: 0 0 8px 0; background: linear-gradient(135deg, #f59e0b, #d97706); -webkit-background-clip: text; -webkit-text-fill-color: transparent;">
              Continue This Conversation
            </h2>
            <p style="color: rgba(255,255,255,0.6); font-size: 14px; margin: 0;">
              Take your ${userMessages.length} messages to another AI
            </p>
          </div>

          <div style="margin-bottom: 20px;">
            <div style="font-size: 12px; color: rgba(255,255,255,0.5); margin-bottom: 12px; text-transform: uppercase; letter-spacing: 0.5px;">
              Choose Your Destination
            </div>

            ${otherLLMs
              .map((llm) => {
                const profile = LLM_PROFILES[llm];
                const userPref = llmPreferences[llm];
                const userUsesFor = userPref?.topTasks?.slice(0, 2).join(', ') || '';

                return `
                <div class="llm-option" data-llm="${llm}" data-url="${profile.url}" style="
                  padding: 16px;
                  background: rgba(255,255,255,0.05);
                  border: 1px solid rgba(255,255,255,0.1);
                  border-radius: 12px;
                  margin-bottom: 10px;
                  cursor: pointer;
                  transition: all 0.2s;
                ">
                  <div style="display: flex; align-items: start; gap: 12px;">
                    <div style="font-size: 28px;">${profile.icon}</div>
                    <div style="flex: 1;">
                      <div style="font-size: 16px; font-weight: 600; margin-bottom: 4px;">${llm}</div>
                      <div style="font-size: 12px; color: rgba(217, 119, 6, 0.9); margin-bottom: 6px;">
                        Best for: ${profile.strengths.slice(0, 3).join(' • ')}
                      </div>
                      ${
                        userUsesFor
                          ? `
                        <div style="font-size: 11px; color: rgba(139, 92, 246, 0.9);">
                          📊 You use this for: ${userUsesFor}
                        </div>
                      `
                          : ''
                      }
                    </div>
                    <div style="color: rgba(255,255,255,0.3); font-size: 20px;">→</div>
                  </div>
                </div>
              `;
              })
              .join('')}
          </div>

          <div style="
            background: rgba(245, 158, 11, 0.1);
            border-radius: 10px;
            padding: 14px;
            margin-bottom: 20px;
            border: 1px solid rgba(245, 158, 11, 0.2);
          ">
            <div style="font-size: 13px; color: rgba(255,255,255,0.8);">
              <strong>What happens:</strong>
              <ul style="margin: 8px 0 0 0; padding-left: 18px; color: rgba(255,255,255,0.7);">
                <li>Your conversation summary is copied to clipboard</li>
                <li>Opens the selected AI in a new tab</li>
                <li>Paste to continue where you left off</li>
              </ul>
            </div>
          </div>

          <div style="display: flex; gap: 12px;">
            <button id="haiven-copy-only" style="
              flex: 1;
              background: rgba(255,255,255,0.1);
              color: white;
              border: none;
              border-radius: 10px;
              padding: 12px;
              font-size: 14px;
              font-weight: 600;
              cursor: pointer;
            ">
              📋 Copy Only
            </button>
            <button id="haiven-close-continue" style="
              flex: 1;
              background: rgba(255,255,255,0.05);
              color: rgba(255,255,255,0.6);
              border: none;
              border-radius: 10px;
              padding: 12px;
              font-size: 14px;
              font-weight: 600;
              cursor: pointer;
            ">
              Cancel
            </button>
          </div>
        </div>
      `;

      document.body.appendChild(modal);

      // Format conversation for export
      const formatConversationForExport = () => {
        let exported = `📝 **Continued Conversation from ${currentPlatform.name}**\n\n`;
        exported += `I was working on this in ${currentPlatform.name} and want to continue here.\n\n`;
        exported += `---\n\n`;
        exported += `**Previous conversation summary:**\n\n`;

        // Interleave user and assistant messages
        const maxMessages = Math.min(userMessages.length, 5); // Limit to last 5 exchanges
        for (let i = 0; i < maxMessages; i++) {
          if (userMessages[i]) {
            exported += `**Me:** ${userMessages[i].substring(0, 300)}${userMessages[i].length > 300 ? '...' : ''}\n\n`;
          }
          if (assistantMessages[i]) {
            exported += `**AI:** ${assistantMessages[i].substring(0, 500)}${assistantMessages[i].length > 500 ? '...' : ''}\n\n`;
          }
        }

        exported += `---\n\n`;
        exported += `Please continue helping me with this task.\n`;

        return exported;
      };

      // Event handlers
      document.querySelectorAll('.llm-option').forEach((option) => {
        option.addEventListener('mouseenter', () => {
          option.style.background = 'rgba(245, 158, 11, 0.15)';
          option.style.borderColor = 'rgba(245, 158, 11, 0.4)';
        });
        option.addEventListener('mouseleave', () => {
          option.style.background = 'rgba(255,255,255,0.05)';
          option.style.borderColor = 'rgba(255,255,255,0.1)';
        });
        option.addEventListener('click', async () => {
          const targetLLM = option.dataset.llm;
          const url = option.dataset.url;

          // Copy conversation
          const exported = formatConversationForExport();
          await navigator.clipboard.writeText(exported);

          // Track the LLM switch
          await fetch(`${API_BASE}/api/llm/track-switch`, {
            method: 'POST',
            headers: await getAuthHeaders(),
            body: JSON.stringify({
              userId: USER_ID,
              fromLLM: currentPlatform.name,
              toLLM: targetLLM,
              messageCount: userMessages.length
            })
          }).catch(() => {});

          // Open target LLM
          window.open(url, '_blank');

          modal.remove();
          showNotification(`✅ Conversation copied! Opening ${targetLLM}...`, 'success');
        });
      });

      document.getElementById('haiven-copy-only').addEventListener('click', async () => {
        const exported = formatConversationForExport();
        await navigator.clipboard.writeText(exported);
        modal.remove();
        showNotification('📋 Conversation copied to clipboard!', 'success');
      });

      document
        .getElementById('haiven-close-continue')
        .addEventListener('click', () => modal.remove());

      // Close on backdrop click
      modal.addEventListener('click', (e) => {
        if (e.target === modal) modal.remove();
      });
    }

    // Show Recall Modal - Superhuman Memory Search
    async function showRecallModal() {
      // Premium CSS variables (matching panel design)
      const cssVars = {
        amber: '#f59e0b',
        amberDark: '#d97706',
        gradient: 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)',
        glow: 'rgba(245, 158, 11, 0.5)',
        bgPrimary: 'rgba(17, 17, 23, 0.98)',
        bgSecondary: 'rgba(255, 255, 255, 0.03)',
        bgTertiary: 'rgba(255, 255, 255, 0.06)',
        bgHover: 'rgba(245, 158, 11, 0.08)',
        textPrimary: 'rgba(255, 255, 255, 0.95)',
        textSecondary: 'rgba(255, 255, 255, 0.6)',
        textTertiary: 'rgba(255, 255, 255, 0.4)',
        borderSubtle: 'rgba(255, 255, 255, 0.08)',
        borderAccent: 'rgba(245, 158, 11, 0.3)',
        spring: 'cubic-bezier(0.34, 1.56, 0.64, 1)'
      };

      // Platform icons for provenance
      const platformIcons = {
        ChatGPT: '🟢',
        Claude: '🟠',
        Grok: '⚪',
        Perplexity: '🔵',
        Gemini: '🔷',
        Unknown: '💭'
      };

      // World icons
      const worldIcons = {
        Work: '💼',
        Tech: '💻',
        Personal: '🏠',
        Health: '🏥',
        Finance: '💰',
        Learning: '📚',
        Creative: '🎨',
        Social: '👥'
      };

      const modal = document.createElement('div');
      modal.id = 'haiven-recall-modal';
      modal.style.cssText = `
        position: fixed;
        top: 50%;
        left: 50%;
        transform: translate(-50%, -50%);
        width: 500px;
        max-height: 70vh;
        background: ${cssVars.bgPrimary};
        backdrop-filter: blur(40px) saturate(180%);
        -webkit-backdrop-filter: blur(40px) saturate(180%);
        border: 1px solid ${cssVars.borderAccent};
        border-radius: 20px;
        box-shadow: 0 0 0 1px rgba(255,255,255,0.05), 0 25px 80px -10px rgba(0,0,0,0.8), 0 0 120px rgba(31,182,255,0.15);
        z-index: 999999;
        font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'Segoe UI', Roboto, sans-serif;
        color: ${cssVars.textPrimary};
        overflow: hidden;
        animation: haivenPanelEntrance 0.4s ${cssVars.spring} forwards;
      `;

      // Build modal with safe DOM methods
      // Header
      const header = createElement(
        'div',
        {
          style: {
            padding: '20px 24px',
            borderBottom: `1px solid ${cssVars.borderSubtle}`,
            display: 'flex',
            alignItems: 'center',
            gap: '12px'
          }
        },
        [
          createElement('span', { textContent: '🔍', style: { fontSize: '24px' } }),
          createElement('div', {}, [
            createElement('div', {
              textContent: 'Recall',
              style: { fontSize: '18px', fontWeight: '700', letterSpacing: '-0.3px' }
            }),
            createElement('div', {
              textContent: 'Search your memories across all platforms',
              style: { fontSize: '12px', color: cssVars.textTertiary, marginTop: '2px' }
            })
          ])
        ]
      );

      // Search input container
      const searchContainer = createElement(
        'div',
        {
          style: { padding: '16px 24px', borderBottom: `1px solid ${cssVars.borderSubtle}` }
        },
        [
          createElement('input', {
            type: 'text',
            id: 'recall-search-input',
            placeholder: 'Search your memories...',
            style: {
              width: '100%',
              padding: '14px 16px',
              background: cssVars.bgTertiary,
              border: `1px solid ${cssVars.borderSubtle}`,
              borderRadius: '12px',
              color: cssVars.textPrimary,
              fontSize: '15px',
              outline: 'none',
              transition: 'all 0.2s ease'
            }
          })
        ]
      );

      // Predictive suggestions section (shows before search)
      const predictiveSection = createElement(
        'div',
        {
          id: 'recall-predictive',
          style: { padding: '16px 24px', borderBottom: `1px solid ${cssVars.borderSubtle}` }
        },
        [
          createElement('div', {
            textContent: '✨ You might need...',
            style: {
              fontSize: '11px',
              fontWeight: '600',
              color: cssVars.amber,
              marginBottom: '12px',
              textTransform: 'uppercase',
              letterSpacing: '0.5px'
            }
          }),
          createElement('div', {
            id: 'recall-predictive-items',
            style: { display: 'flex', flexDirection: 'column', gap: '8px' }
          })
        ]
      );

      // Results container
      const resultsContainer = createElement(
        'div',
        {
          id: 'recall-results',
          style: {
            padding: '16px 24px',
            maxHeight: '300px',
            overflowY: 'auto'
          }
        },
        [
          createElement('div', {
            id: 'recall-results-content',
            textContent: 'Start typing to search...',
            style: { color: cssVars.textTertiary, textAlign: 'center', padding: '20px' }
          })
        ]
      );

      // Footer with keyboard hints
      const footer = createElement(
        'div',
        {
          style: {
            padding: '12px 24px',
            borderTop: `1px solid ${cssVars.borderSubtle}`,
            display: 'flex',
            justifyContent: 'space-between',
            alignItems: 'center',
            fontSize: '11px',
            color: cssVars.textTertiary
          }
        },
        [
          createElement('span', { textContent: 'Click to insert into conversation' }),
          createElement('span', { textContent: 'ESC to close' })
        ]
      );

      modal.appendChild(header);
      modal.appendChild(searchContainer);
      modal.appendChild(predictiveSection);
      modal.appendChild(resultsContainer);
      modal.appendChild(footer);

      document.body.appendChild(modal);

      // Focus search input
      const searchInput = document.getElementById('recall-search-input');
      setTimeout(() => searchInput.focus(), 100);

      // Style search input on focus
      searchInput.addEventListener('focus', () => {
        searchInput.style.borderColor = cssVars.borderAccent;
        searchInput.style.boxShadow = `0 0 0 3px ${cssVars.bgHover}`;
      });
      searchInput.addEventListener('blur', () => {
        searchInput.style.borderColor = cssVars.borderSubtle;
        searchInput.style.boxShadow = 'none';
      });

      // Load predictive suggestions on open
      loadPredictiveSuggestions();

      // Debounced search
      let searchTimeout;
      searchInput.addEventListener('input', (e) => {
        clearTimeout(searchTimeout);
        const query = e.target.value.trim();

        if (query.length < 2) {
          document.getElementById('recall-results-content').textContent =
            'Start typing to search...';
          document.getElementById('recall-predictive').style.display = 'block';
          return;
        }

        document.getElementById('recall-predictive').style.display = 'none';
        document.getElementById('recall-results-content').textContent = 'Searching...';

        searchTimeout = setTimeout(() => searchMemories(query), 300);
      });

      // Close on ESC
      const handleEsc = (e) => {
        if (e.key === 'Escape') {
          modal.remove();
          document.removeEventListener('keydown', handleEsc);
        }
      };
      document.addEventListener('keydown', handleEsc);

      // Close on backdrop click
      modal.addEventListener('click', (e) => {
        if (e.target === modal) {
          modal.remove();
          document.removeEventListener('keydown', handleEsc);
        }
      });

      // Load predictive suggestions based on current context
      async function loadPredictiveSuggestions() {
        try {
          // Get current conversation context for prediction
          const inputArea = document.querySelector(currentPlatform.inputSelector);
          const currentInput = inputArea?.value || inputArea?.textContent || '';

          const response = await safeFetch(
            `${API_BASE}/api/memory/recall`,
            {
              method: 'POST',
              headers: await getAuthHeaders(),
              body: JSON.stringify({
                userId: USER_ID,
                query: currentInput || 'recent important memories',
                limit: 3,
                predictive: true
              })
            },
            5000
          );

          const data = await response.json();
          const predictiveItems = document.getElementById('recall-predictive-items');
          predictiveItems.textContent = '';

          if (data.success && data.memories && data.memories.length > 0) {
            data.memories.forEach((memory) => {
              const item = createMemoryCard(memory, true);
              predictiveItems.appendChild(item);
            });
          } else {
            predictiveItems.appendChild(
              createElement('div', {
                textContent: 'No predictions available yet',
                style: { color: cssVars.textTertiary, fontSize: '13px' }
              })
            );
          }
        } catch (_error) {
          // Silently fail - predictive is optional
          document.getElementById('recall-predictive').style.display = 'none';
        }
      }

      // Search memories
      async function searchMemories(query) {
        try {
          const response = await safeFetch(
            `${API_BASE}/api/memory/recall`,
            {
              method: 'POST',
              headers: await getAuthHeaders(),
              body: JSON.stringify({
                userId: USER_ID,
                query: query,
                limit: 10
              })
            },
            8000
          );

          const data = await response.json();
          const resultsContent = document.getElementById('recall-results-content');
          resultsContent.textContent = '';

          if (data.success && data.memories && data.memories.length > 0) {
            data.memories.forEach((memory) => {
              const card = createMemoryCard(memory, false);
              resultsContent.appendChild(card);
            });
          } else {
            resultsContent.appendChild(
              createElement(
                'div',
                {
                  style: { textAlign: 'center', padding: '30px', color: cssVars.textTertiary }
                },
                [
                  createElement('div', {
                    textContent: '🔍',
                    style: { fontSize: '32px', marginBottom: '12px' }
                  }),
                  createElement('div', { textContent: 'No memories found' }),
                  createElement('div', {
                    textContent: 'Try different keywords',
                    style: { fontSize: '12px', marginTop: '4px' }
                  })
                ]
              )
            );
          }
        } catch (error) {
          console.error('Recall search error:', error);
          document.getElementById('recall-results-content').textContent =
            'Search failed. Please try again.';
        }
      }

      // Create memory card with cross-platform provenance
      function createMemoryCard(memory, isPredictive) {
        const platform = memory.metadata?.platform || memory.platform || 'Unknown';
        const world = memory.world || memory.metadata?.world || null;
        const timestamp = memory.created_at || memory.timestamp;
        const relevance = memory.relevance_score || memory.score;

        const card = createElement(
          'div',
          {
            className: 'recall-memory-card',
            style: {
              padding: '14px 16px',
              background: isPredictive ? cssVars.bgHover : cssVars.bgSecondary,
              border: `1px solid ${isPredictive ? cssVars.borderAccent : cssVars.borderSubtle}`,
              borderRadius: '12px',
              marginBottom: '10px',
              cursor: 'pointer',
              transition: `all 0.2s ${cssVars.spring}`
            }
          },
          [
            // Provenance row (platform + world + time)
            createElement(
              'div',
              {
                style: {
                  display: 'flex',
                  alignItems: 'center',
                  gap: '8px',
                  marginBottom: '8px',
                  flexWrap: 'wrap'
                }
              },
              [
                // Platform badge
                createElement(
                  'span',
                  {
                    style: {
                      display: 'inline-flex',
                      alignItems: 'center',
                      gap: '4px',
                      fontSize: '11px',
                      color: cssVars.amber,
                      background: 'rgba(245, 158, 11, 0.1)',
                      padding: '3px 8px',
                      borderRadius: '6px'
                    }
                  },
                  [
                    createElement('span', {
                      textContent: platformIcons[platform] || platformIcons.Unknown
                    }),
                    createElement('span', { textContent: platform })
                  ]
                ),
                // World badge (if available)
                world
                  ? createElement(
                      'span',
                      {
                        style: {
                          display: 'inline-flex',
                          alignItems: 'center',
                          gap: '4px',
                          fontSize: '11px',
                          color: cssVars.amberDark,
                          background: 'rgba(217, 119, 6, 0.1)',
                          padding: '3px 8px',
                          borderRadius: '6px'
                        }
                      },
                      [
                        createElement('span', { textContent: worldIcons[world] || '📁' }),
                        createElement('span', { textContent: world })
                      ]
                    )
                  : null,
                // Timestamp
                createElement('span', {
                  textContent: formatTimeAgo(timestamp),
                  style: { fontSize: '11px', color: cssVars.textTertiary, marginLeft: 'auto' }
                })
              ].filter(Boolean)
            ),
            // Content
            createElement('div', {
              textContent:
                (memory.content || '').substring(0, 150) +
                ((memory.content || '').length > 150 ? '...' : ''),
              style: { fontSize: '13px', lineHeight: '1.5', color: cssVars.textSecondary }
            }),
            // Relevance indicator (for search results)
            relevance
              ? createElement(
                  'div',
                  {
                    style: { marginTop: '8px', display: 'flex', alignItems: 'center', gap: '6px' }
                  },
                  [
                    createElement(
                      'div',
                      {
                        style: {
                          width: '40px',
                          height: '3px',
                          background: cssVars.bgTertiary,
                          borderRadius: '2px',
                          overflow: 'hidden'
                        }
                      },
                      [
                        createElement('div', {
                          style: {
                            width: `${Math.min(relevance * 100, 100)}%`,
                            height: '100%',
                            background: cssVars.gradient
                          }
                        })
                      ]
                    ),
                    createElement('span', {
                      textContent: `${Math.round(relevance * 100)}% match`,
                      style: { fontSize: '10px', color: cssVars.textTertiary }
                    })
                  ]
                )
              : null
          ].filter(Boolean)
        );

        // Hover effects
        card.addEventListener('mouseenter', () => {
          card.style.background = cssVars.bgHover;
          card.style.borderColor = cssVars.borderAccent;
          card.style.transform = 'translateX(4px)';
        });
        card.addEventListener('mouseleave', () => {
          card.style.background = isPredictive ? cssVars.bgHover : cssVars.bgSecondary;
          card.style.borderColor = isPredictive ? cssVars.borderAccent : cssVars.borderSubtle;
          card.style.transform = 'translateX(0)';
        });

        // Click to inject
        card.addEventListener('click', () => {
          injectMemoryIntoConversation(memory);
          modal.remove();
          document.removeEventListener('keydown', handleEsc);
        });

        return card;
      }

      // Inject memory into conversation
      function injectMemoryIntoConversation(memory) {
        const inputArea = document.querySelector(currentPlatform.inputSelector);
        if (!inputArea) {
          showNotification('❌ Could not find input area', 'error');
          return;
        }

        const platform = memory.metadata?.platform || memory.platform || 'Unknown';
        const world = memory.world || memory.metadata?.world || '';

        // Format memory for injection based on target platform
        let formattedMemory;
        if (currentPlatform.name === 'Claude') {
          // XML format for Claude
          formattedMemory = `<recalled_memory source="${platform}" world="${world}">\n${memory.content}\n</recalled_memory>\n\nBased on this memory: `;
        } else {
          // Natural format for others
          formattedMemory = `[Recalled from ${platform}${world ? ` (${world})` : ''}]\n"${memory.content}"\n\nConsidering this: `;
        }

        const currentText = inputArea.value || inputArea.textContent || '';
        const newText = formattedMemory + currentText;

        if (inputArea.tagName === 'TEXTAREA' || inputArea.tagName === 'INPUT') {
          inputArea.value = newText;
        } else {
          inputArea.textContent = newText;
        }

        inputArea.dispatchEvent(new Event('input', { bubbles: true }));
        inputArea.focus();

        showNotification(`✨ Memory recalled from ${platform}`, 'success');
      }
    }

    // Extract conversation context from the current page
    function extractConversationContext() {
      try {
        // Get the current input field value (what user is typing or might type)
        const inputField = document.querySelector(currentPlatform.inputSelector);
        let currentInput = '';
        if (inputField) {
          currentInput = inputField.value || inputField.textContent || '';
        }

        // Get recent messages from the conversation
        const messageElements = document.querySelectorAll(currentPlatform.messageSelector);
        const messages = [];

        // Get last 3-5 messages to understand context
        const recentMessages = Array.from(messageElements).slice(-5);
        for (const msg of recentMessages) {
          const text = msg.textContent || msg.innerText || '';
          if (text.trim().length > 10) {
            messages.push(text.trim());
          }
        }

        // Combine input + recent messages into context
        const conversationContext = [currentInput, ...messages]
          .filter((text) => text.length > 0)
          .join(' ')
          .substring(0, 500); // Limit to 500 chars for query

        return conversationContext || 'recent context';
      } catch (error) {
        console.error('Failed to extract conversation context:', error);
        return 'recent context';
      }
    }

    // Show notification toast
    function showNotification(message, type = 'info') {
      // Remove any existing notification
      const existing = document.getElementById('haiven-save-toast');
      if (existing) existing.remove();

      const icons = { success: '✓', error: '✕', warning: '⚠', info: 'ℹ' };
      const colors = {
        success: { border: '#22c55e', bg: 'rgba(34, 197, 94, 0.15)', icon: '#22c55e' },
        error: { border: '#ef4444', bg: 'rgba(239, 68, 68, 0.15)', icon: '#ef4444' },
        warning: { border: '#f59e0b', bg: 'rgba(245, 158, 11, 0.15)', icon: '#f59e0b' },
        info: { border: '#6366f1', bg: 'rgba(99, 102, 241, 0.15)', icon: '#6366f1' }
      };
      const color = colors[type] || colors.info;

      const notification = document.createElement('div');
      notification.id = 'haiven-save-toast';
      notification.style.cssText = `
        position: fixed;
        bottom: 24px;
        left: 24px;
        z-index: 9999999;
        min-width: 280px;
        background: rgba(17, 17, 23, 0.98);
        backdrop-filter: blur(20px);
        border-radius: 10px;
        border-left: 3px solid ${color.border};
        padding: 14px 16px;
        display: flex;
        align-items: center;
        gap: 12px;
        box-shadow: 0 16px 32px rgba(0, 0, 0, 0.4);
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        animation: haivenToastIn 0.3s cubic-bezier(0.4, 0, 0.2, 1);
      `;

      // Add animation keyframes if not present
      if (!document.getElementById('haiven-toast-keyframes')) {
        const style = document.createElement('style');
        style.id = 'haiven-toast-keyframes';
        style.textContent = `
          @keyframes haivenToastIn {
            from { opacity: 0; transform: translateY(10px) scale(0.95); }
            to { opacity: 1; transform: translateY(0) scale(1); }
          }
          @keyframes haivenToastOut {
            to { opacity: 0; transform: translateY(10px) scale(0.95); }
          }
        `;
        document.head.appendChild(style);
      }

      // Icon container
      const iconEl = document.createElement('div');
      iconEl.style.cssText = `
        width: 32px;
        height: 32px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        border-radius: 8px;
        background: ${color.bg};
        color: ${color.icon};
        font-weight: bold;
      `;
      iconEl.textContent = icons[type] || icons.info;

      // Content
      const content = document.createElement('div');
      content.style.cssText = 'flex: 1;';

      const title = document.createElement('div');
      title.style.cssText =
        'font-size: 13px; font-weight: 600; color: #ffffff; margin-bottom: 2px;';
      title.textContent = type === 'success' ? 'Saved to Haiven' : message;

      const subtitle = document.createElement('div');
      subtitle.style.cssText = 'font-size: 12px; color: rgba(255, 255, 255, 0.6);';
      subtitle.textContent = type === 'success' ? 'Memory added to your collection' : '';

      content.appendChild(title);
      if (type === 'success') content.appendChild(subtitle);

      notification.appendChild(iconEl);
      notification.appendChild(content);
      document.body.appendChild(notification);

      setTimeout(() => {
        notification.style.animation = 'haivenToastOut 0.25s ease forwards';
        setTimeout(() => notification.remove(), 250);
      }, 3000);
    }

    // Helper functions
    function escapeHtml(text) {
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    }

    function formatTimeAgo(timestamp) {
      const date = new Date(timestamp);
      const now = new Date();
      const seconds = Math.floor((now - date) / 1000);

      if (seconds < 60) return 'just now';
      if (seconds < 3600) return `${Math.floor(seconds / 60)}m ago`;
      if (seconds < 86400) return `${Math.floor(seconds / 3600)}h ago`;
      if (seconds < 604800) return `${Math.floor(seconds / 86400)}d ago`;
      return date.toLocaleDateString();
    }

    // Add CSS animations
    const style = document.createElement('style');
    style.textContent = `
    @keyframes slideIn {
      from {
        transform: translateX(400px);
        opacity: 0;
      }
      to {
        transform: translateX(0);
        opacity: 1;
      }
    }

    @keyframes slideOut {
      from {
        transform: translateX(0);
        opacity: 1;
      }
      to {
        transform: translateX(400px);
        opacity: 0;
      }
    }
  `;
    document.head.appendChild(style);

    // ========================================
    // AUTO-CAPTURE FUNCTIONALITY
    // ========================================

    const lastProcessedMessages = new Set();
    let observerActive = false;

    // Start auto-capture monitoring
    async function startAutoCapture() {
      try {
        const result = await browserAPI.storage.local.get(['autoCapture']);
        if (result.autoCapture !== false) {
          console.log('🧠 Haiven: Auto-capture enabled');
          observeConversation();
          processExistingMessages();
        } else {
          console.log('🧠 Haiven: Auto-capture disabled');
        }
      } catch (error) {
        console.error('Failed to check auto-capture setting:', error);
      }
    }

    // Monitor for new messages
    function observeConversation() {
      if (observerActive) return;

      // Wait for conversation container
      const checkForContainer = cleanupManager.addInterval(
        setInterval(() => {
          const container = findConversationContainer();
          if (container) {
            clearInterval(checkForContainer);
            setupObserver(container);
          }
        }, 1000)
      );

      // Stop checking after 10 seconds
      setTimeout(() => clearInterval(checkForContainer), 10000);
    }

    // Find conversation container based on platform
    function findConversationContainer() {
      if (currentPlatform.name === 'ChatGPT') {
        return document.querySelector('[role="presentation"]') || document.querySelector('main');
      }
      if (currentPlatform.name === 'Claude') {
        return (
          document.querySelector('[data-testid="conversation"]') || document.querySelector('main')
        );
      }
      // Fallback for other platforms
      return document.body;
    }

    // Setup mutation observer with debouncing
    let observerDebounceTimer = null;
    function setupObserver(container) {
      const observer = new MutationObserver(() => {
        // Debounce: Wait 500ms after last mutation before processing
        // This prevents capturing the same streaming message multiple times
        if (observerDebounceTimer) {
          clearTimeout(observerDebounceTimer);
        }
        observerDebounceTimer = setTimeout(() => {
          processNewMessages();
        }, 500);
      });

      observer.observe(container, {
        childList: true,
        subtree: true
      });

      // Register for cleanup
      cleanupManager.addObserver(observer);

      observerActive = true;
      console.log('🧠 Haiven: Observer active on', currentPlatform.name);
    }

    // Process existing messages on page load (uses same deduplication as processNewMessages)
    function processExistingMessages() {
      setTimeout(() => {
        console.log('🧠 Haiven: Processing existing messages on page load');
        processNewMessages(); // Reuse the deduplication logic
      }, 2000); // Wait for page to fully load
    }

    // P0.3: Process newly added messages with conversation context
    // This is the SINGLE function that handles all message captures with deduplication
    function processNewMessages() {
      const messages = getAllMessages();

      messages.forEach((msg) => {
        const msgData = extractMessageData(msg);
        if (!msgData) return;

        const msgId = getMessageId(msgData);

        // Skip if already processed
        if (lastProcessedMessages.has(msgId)) {
          return;
        }

        // Track conversation context
        if (msgData.isUser) {
          // Store user question for context
          lastUserMessage = msgData.content;
        } else {
          // Assistant response - associate with user question
          msgData.contextQuestion = lastUserMessage;
        }

        // Mark as processed BEFORE capturing to prevent race conditions
        lastProcessedMessages.add(msgId);

        if (shouldCapture(msgData)) {
          captureMessage(msgData);
        }
      });
    }

    // Get all message elements
    function getAllMessages() {
      const userSelector = currentPlatform.messageSelector;
      const { assistantSelector } = currentPlatform;

      const userMessages = Array.from(document.querySelectorAll(userSelector));
      const assistantMessages = Array.from(document.querySelectorAll(assistantSelector));

      return [...userMessages, ...assistantMessages];
    }

    // Extract message data from element
    function extractMessageData(element) {
      const content = element.textContent?.trim();
      if (!content || content.length < 10) return null;

      // Determine if user or assistant
      const isUser =
        element.matches(currentPlatform.messageSelector) ||
        element.getAttribute('data-message-author-role') === 'user' ||
        element.classList.contains('user-message') ||
        element.classList.contains('font-user-message');

      return {
        content,
        isUser,
        element
      };
    }

    // Generate unique message ID
    function getMessageId(msgData) {
      const role = msgData.isUser ? 'user' : 'assistant';
      // Use element position or first 100 chars for stable ID (NOT content.length which changes during streaming)
      const elementId =
        msgData.element?.getAttribute('data-message-id') ||
        msgData.element?.getAttribute('data-testid') ||
        '';
      // Use first 100 chars normalized (stable during streaming after initial content)
      const contentKey = msgData.content.substring(0, 100).replace(/\s+/g, ' ').trim();
      return `${role}-${elementId}-${contentKey}`;
    }

    // P0.2: Detect high-value responses worth saving
    function isHighValueResponse(content) {
      const wordCount = content.split(/\s+/).length;

      // Long-form content (>500 words)
      if (wordCount > 500) return true;

      // Structured content (phases, steps, numbered lists)
      if (/\b(phase|step|stage|part)\s+\d+/i.test(content)) return true;
      if (/\d+\.\s+.+\n.*\d+\./s.test(content)) return true; // Multi-line numbered list

      // Educational content
      if (/(roadmap|tutorial|guide|how to|learn .+ in)/i.test(content)) return true;

      // Code examples
      if ((content.match(/```/g) || []).length >= 2) return true; // Has code blocks

      // Lists with multiple items (>5 bullet points)
      if ((content.match(/^[-•*]\s+/gm) || []).length > 5) return true;

      return false;
    }

    // Determine if message should be captured
    function shouldCapture(msgData) {
      const { content, isUser } = msgData;

      // EXPLICIT CAPTURE: User says "Haiven, remember..." or "Remember that..."
      if (isUser) {
        const explicitPatterns = [
          /haiven,?\s+(remember|save|store|capture)/i,
          /^remember\s+(that|this)/i,
          /^save\s+(this|that)/i,
          /^store\s+(this|that)/i
        ];

        return explicitPatterns.some((pattern) => pattern.test(content));
      }

      // P0.2: HIGH-VALUE AUTO-CAPTURE - Show save suggestion for valuable responses
      if (!isUser && isHighValueResponse(content)) {
        // Show save suggestion instead of auto-capturing silently
        showAutoSaveSuggestion(content, 0.85); // High confidence for structured content
        return false; // Don't auto-capture, let user decide
      }

      // AUTO-CAPTURE: Assistant responses with important keywords
      if (!isUser && content.length >= 100) {
        const importantPatterns = [
          /remember/i,
          /important/i,
          /key point/i,
          /takeaway/i,
          /summary/i,
          /action item/i,
          /follow up/i,
          /deadline/i,
          /meeting/i,
          /decision/i,
          /appointment/i,
          /scheduled/i,
          /\d{1,2}:\d{2}\s*(am|pm)/i, // Time mentions
          /\d{1,2}\/\d{1,2}\/\d{2,4}/, // Date mentions
          /next (week|month|monday|tuesday|wednesday|thursday|friday)/i
        ];

        return importantPatterns.some((pattern) => pattern.test(content));
      }

      return false;
    }

    // Detect explicit save triggers (Capital One-style explicit triggers)
    function detectSaveTrigger(text) {
      const SAVE_TRIGGERS = [
        'remember this',
        'save this',
        'haiven remember',
        'haiven save',
        'i want to remember',
        "don't forget",
        'make a note',
        'note to self',
        'remember:',
        'important:',
        'save:',
        'note:'
      ];

      const lowerText = text.toLowerCase();

      for (const trigger of SAVE_TRIGGERS) {
        if (lowerText.includes(trigger)) {
          // Extract content after trigger
          const parts = text.split(new RegExp(trigger, 'i'));
          const contentAfterTrigger = parts.length > 1 ? parts[1].trim() : text;

          return {
            triggered: true,
            trigger: trigger,
            content: contentAfterTrigger || text,
            confidence: 1.0 // Explicit = 100% confidence
          };
        }
      }

      return { triggered: false, content: text };
    }

    // Capture message via background script
    function captureMessage(msgData) {
      const { content, isUser } = msgData;

      // Privacy filter: Block if contains sensitive data
      if (containsSensitiveData(content)) {
        console.warn('🔒 Haiven: Skipped capture - sensitive data detected');
        showNotification('🔒 Sensitive data detected - not captured', 'warning');
        return;
      }

      // Check for explicit save triggers
      const triggerCheck = detectSaveTrigger(content);
      const shouldBypassFilter = triggerCheck.triggered;

      // Sanitize content before capture (additional safety layer)
      const sanitizedContent = sanitizeContent(triggerCheck.content);

      // OPTIMISTIC UI: Show instant loading feedback
      const loadingToast = showLoadingToast(
        shouldBypassFilter ? '✓ Saving to Haiven...' : 'Analyzing memory...'
      );

      // Log trigger detection
      if (shouldBypassFilter) {
        console.log(`🎯 Haiven: Explicit trigger detected - "${triggerCheck.trigger}"`);
      }

      // P0.3: Include conversation context
      browserAPI.runtime.sendMessage(
        {
          action: 'decompose-conversation',
          content: sanitizedContent,
          source: currentPlatform.name,
          platform: currentPlatform.name,
          url: window.location.href,
          explicit: isUser,
          bypassFilter: shouldBypassFilter, // Pass bypass flag to backend
          trigger: triggerCheck.trigger, // Pass trigger phrase for logging
          contextQuestion: msgData.contextQuestion || null // P0.3: Include user's question
        },
        (response) => {
          // Remove loading toast
          if (loadingToast) loadingToast.remove();

          if (response && response.success) {
            console.log('🧠 Haiven: Captured -', sanitizedContent.substring(0, 60));
            // Always show success notification for explicit triggers
            if (shouldBypassFilter || isUser) {
              showCaptureNotification();
            }
          } else {
            console.error('🧠 Haiven: Capture failed:', response?.error);
            showErrorNotification('Failed to save memory');
          }
        }
      );
    }

    // Show visual notification when memory is captured - PROMINENT bottom-left toast
    function showCaptureNotification(message = 'Saved to Haiven', confidence = null) {
      // Remove any existing notification
      const existing = document.getElementById('haiven-save-notification');
      if (existing) existing.remove();

      const notification = document.createElement('div');
      notification.id = 'haiven-save-notification';

      // Add CSS animations if not already added
      if (!document.getElementById('haiven-save-styles')) {
        const style = document.createElement('style');
        style.id = 'haiven-save-styles';
        style.textContent = `
          @keyframes haiven-slide-in {
            from { opacity: 0; transform: translateX(-100%); }
            to { opacity: 1; transform: translateX(0); }
          }
          @keyframes haiven-slide-out {
            from { opacity: 1; transform: translateX(0); }
            to { opacity: 0; transform: translateX(-100%); }
          }
          @keyframes haiven-checkmark {
            0% { transform: scale(0); }
            50% { transform: scale(1.2); }
            100% { transform: scale(1); }
          }
        `;
        document.head.appendChild(style);
      }

      notification.style.cssText = `
        position: fixed;
        bottom: 24px;
        left: 24px;
        background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        color: white;
        padding: 16px 24px;
        border-radius: 12px;
        font-size: 15px;
        font-weight: 600;
        z-index: 9999999;
        box-shadow: 0 8px 32px rgba(16, 185, 129, 0.4), 0 0 0 1px rgba(255, 255, 255, 0.1);
        animation: haiven-slide-in 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
        display: flex;
        align-items: center;
        gap: 12px;
        max-width: 320px;
      `;

      // Build content with checkmark
      const checkmark = document.createElement('div');
      checkmark.style.cssText = `
        width: 28px;
        height: 28px;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        animation: haiven-checkmark 0.5s ease-out 0.2s both;
      `;
      checkmark.textContent = '✓';

      const textContainer = document.createElement('div');

      const title = document.createElement('div');
      title.style.cssText = 'font-weight: 700; margin-bottom: 2px;';
      title.textContent = message;

      const subtitle = document.createElement('div');
      subtitle.style.cssText = 'font-size: 12px; opacity: 0.9; font-weight: 400;';
      subtitle.textContent = confidence
        ? `Memory added (${Math.round(confidence * 100)}% confident)`
        : 'Memory added to your collection';

      textContainer.appendChild(title);
      textContainer.appendChild(subtitle);

      notification.appendChild(checkmark);
      notification.appendChild(textContainer);

      document.body.appendChild(notification);

      // Auto-remove after 3.5 seconds
      setTimeout(() => {
        notification.style.animation = 'haiven-slide-out 0.3s ease-out forwards';
        setTimeout(() => notification.remove(), 300);
      }, 3500);
    }

    function showLoadingToast(message) {
      const toast = document.createElement('div');
      toast.className = 'haiven-loading-toast';
      toast.innerHTML = `
      <div class="spinner"></div>
      <span>${escapeHtml(message)}</span>
    `;
      toast.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: rgba(15, 15, 35, 0.95);
      backdrop-filter: blur(10px);
      color: white;
      padding: 12px 20px;
      border-radius: 12px;
      font-size: 14px;
      font-weight: 500;
      z-index: 9999999;
      box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
      display: flex;
      align-items: center;
      gap: 12px;
      animation: slideIn 0.3s ease-out;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      border: 1px solid rgba(255, 255, 255, 0.1);
    `;

      // Add spinner animation
      const style = document.createElement('style');
      style.textContent = `
      .haiven-loading-toast .spinner {
        width: 16px;
        height: 16px;
        border: 2px solid rgba(255, 255, 255, 0.3);
        border-top-color: white;
        border-radius: 50%;
        animation: spin 0.6s linear infinite;
      }
      @keyframes spin {
        to { transform: rotate(360deg); }
      }
    `;
      if (!document.getElementById('haiven-loading-styles')) {
        style.id = 'haiven-loading-styles';
        document.head.appendChild(style);
      }

      document.body.appendChild(toast);
      return toast;
    }

    function _showDecomposedNotification(count, worlds) {
      const notification = document.createElement('div');
      const worldList = worlds.length > 0 ? worlds.join(', ') : 'Multiple Worlds';
      notification.innerHTML = `
      <div style="display: flex; align-items: center; gap: 8px;">
        <span style="font-size: 18px;">🌍</span>
        <div>
          <div style="font-weight: 600; margin-bottom: 2px;">Memory Decomposed!</div>
          <div style="font-size: 12px; opacity: 0.9;">Split into ${count} memories: ${worldList}</div>
        </div>
      </div>
    `;
      notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: linear-gradient(135deg, #f59e0b, #d97706);
      color: white;
      padding: 14px 20px;
      border-radius: 12px;
      font-size: 14px;
      z-index: 9999999;
      box-shadow: 0 4px 20px rgba(16, 185, 129, 0.4);
      animation: slideIn 0.3s ease-out;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      max-width: 350px;
      border: 1px solid rgba(255, 255, 255, 0.2);
    `;

      document.body.appendChild(notification);

      setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease-out';
        setTimeout(() => notification.remove(), 300);
      }, 4000);
    }

    function showErrorNotification(message) {
      const notification = document.createElement('div');
      notification.innerHTML = `
      <div style="display: flex; align-items: center; gap: 8px;">
        <span style="font-size: 18px;">⚠️</span>
        <span>${message}</span>
      </div>
    `;
      notification.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: linear-gradient(135deg, #EF4444, #DC2626);
      color: white;
      padding: 12px 20px;
      border-radius: 12px;
      font-size: 14px;
      font-weight: 500;
      z-index: 9999999;
      box-shadow: 0 4px 20px rgba(239, 68, 68, 0.4);
      animation: slideIn 0.3s ease-out;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      border: 1px solid rgba(255, 255, 255, 0.2);
    `;

      document.body.appendChild(notification);

      setTimeout(() => {
        notification.style.animation = 'slideOut 0.3s ease-out';
        setTimeout(() => notification.remove(), 300);
      }, 3000);
    }

    // ========================================
    // KEYBOARD SHORTCUT CAPTURE (Cmd+Shift+M)
    // ========================================

    function getSelectedOrRecentText() {
      // First, try to get selected text
      const selection = window.getSelection().toString().trim();
      if (selection) return selection;

      // If no selection, get text from input area
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (inputArea) {
        const text = inputArea.value || inputArea.textContent || '';
        return text.trim();
      }

      return '';
    }

    function quickCapture() {
      console.log('🧠 Haiven: quickCapture() called');

      const text = getSelectedOrRecentText();

      if (!text) {
        console.warn('🧠 Haiven: No text to save');
        showNotification('⚠️ Nothing to save - select text or type in input area', 'warning');
        return;
      }

      if (text.length < 10) {
        console.warn('🧠 Haiven: Text too short:', text.length);
        showNotification('⚠️ Text too short to save (minimum 10 characters)', 'warning');
        return;
      }

      // Check privacy
      if (containsSensitiveData(text)) {
        console.warn('🧠 Haiven: Sensitive data detected');
        showNotification('🔒 Sensitive data detected - not captured', 'warning');
        return;
      }

      // Check USER_ID
      if (!USER_ID) {
        console.error('🧠 Haiven: USER_ID not set!');
        showNotification('❌ User ID not set. Please configure extension.', 'error');
        return;
      }

      console.log('🧠 Haiven: Capturing text:', `${text.substring(0, 50)}...`);

      // Save immediately
      try {
        captureMessage({ content: text, isUser: true });
        showCaptureNotification('⚡ Quick-saved to Haiven');
      } catch (error) {
        console.error('🧠 Haiven: Capture failed:', error);
        showNotification(`❌ Failed to save: ${error.message}`, 'error');
      }
    }

    // Listen for Cmd+Shift+S (or Ctrl+Shift+S on Windows) - S for Save
    document.addEventListener('keydown', (e) => {
      const isShortcut = (e.metaKey || e.ctrlKey) && e.shiftKey && (e.key === 'S' || e.key === 's');

      if (isShortcut) {
        console.log('🧠 Haiven: Keyboard shortcut triggered!', {
          key: e.key,
          metaKey: e.metaKey,
          ctrlKey: e.ctrlKey,
          shiftKey: e.shiftKey,
          userId: USER_ID ? 'SET' : 'NOT SET'
        });

        e.preventDefault();
        e.stopPropagation();

        try {
          quickCapture();
        } catch (error) {
          console.error('🧠 Haiven: Quick capture error:', error);
          showNotification(`❌ Quick capture failed: ${error.message}`, 'error');
        }
      }
    });

    console.log('⌨️ Haiven: Keyboard shortcut ready (Cmd+Shift+S to quick-save)');

    // ========================================
    // CTRL+H - QUICK MEMORY PANEL + CONTEXT INJECTION
    // ========================================

    // Listen for Ctrl+H (or Cmd+H on Mac) - H for Haiven
    document.addEventListener('keydown', (e) => {
      const isHaivenShortcut =
        (e.metaKey || e.ctrlKey) && !e.shiftKey && (e.key === 'H' || e.key === 'h');

      if (isHaivenShortcut) {
        console.log('🧠 Haiven: Ctrl+H shortcut triggered!');

        e.preventDefault();
        e.stopPropagation();

        // If panel is already open, close it
        const existingPanel = document.getElementById('haiven-panel');
        if (existingPanel) {
          existingPanel.remove();
          return;
        }

        // Show panel and optionally auto-inject context based on current input
        showHaivenPanel();

        // If there's text in the input, automatically trigger a memory search
        const inputText = getInputText();
        if (inputText && inputText.length > 10) {
          console.log('🧠 Haiven: Auto-searching for context based on input');
          // Trigger automatic context search after panel opens
          setTimeout(() => {
            autoSearchContext(inputText);
          }, 300);
        }
      }
    });

    // Get current text from the input area
    function getInputText() {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) return '';
      return (inputArea.value || inputArea.textContent || '').trim();
    }

    // Auto-search for relevant memories based on input text
    async function autoSearchContext(queryText) {
      if (!USER_ID) {
        console.warn('🧠 Haiven: Cannot auto-search - no credentials');
        return;
      }

      try {
        showNotification('🔍 Finding relevant memories...', 'info');

        const response = await safeFetch(
          `${API_BASE}/api/memory/recall`,
          {
            method: 'POST',
            headers: {
              'Content-Type': 'application/json',
              Authorization: `Bearer ${USER_ID}`
            },
            body: JSON.stringify({
              userId: USER_ID,
              query: queryText.substring(0, 500),
              limit: 5
            })
          },
          8000
        ); // 8 second timeout

        if (response.ok) {
          const data = await response.json();
          if (data.memories && data.memories.length > 0) {
            showNotification(`✨ Found ${data.memories.length} relevant memories`, 'success');
            // Update the suggestion widget if it exists
            showSuggestionWidget(data.memories);
          } else {
            showNotification('📭 No matching memories found', 'info');
          }
        }
      } catch (error) {
        console.error('🧠 Haiven: Auto-search failed:', error);
      }
    }

    console.log('⌨️ Haiven: Ctrl+H shortcut ready (open panel + auto-search)');

    // ========================================
    // PREDICTIVE AUTO-CAPTURE
    // ========================================

    const HIGH_CONFIDENCE_PATTERNS = [
      // Explicit save requests
      /(haiven,?\s+)?(remember|save|store|keep|note)(\s+this)?[:\s].{10,}/i,
      /don't forget[:\s].{10,}/i,
      /important[:\s].{10,}/i,

      // Preferences and habits
      /I (prefer|like|love|hate|avoid|always|never)\s+.{10,}/i,
      /my (favorite|preferred|usual|default)\s+.{5,}/i,

      // Professional info
      /my (role|job|position|title|company|team) (is|:)\s*.{5,}/i,
      /I (work at|work for|work with|report to)\s+.{5,}/i,
      /(my|our) (tech stack|stack|tools|setup|framework|library|language) (is|are|includes?|uses?|:)\s*.{10,}/i,

      // Projects and work
      /I('m| am) (building|developing|creating|working on|launching)\s+.{10,}/i,
      /(I use|I'm using|we use|we're using)\s+.{10,}/i,

      // Statements about specific topics (like "about texas skill games")
      /about\s+.{10,}\s+(is|are|means?|refers?|involves?)/i,
      /(the|our|my)\s+.{5,}\s+(project|product|app|system|platform)\s+(is|does|has)/i
    ];

    const MEDIUM_CONFIDENCE_PATTERNS = [
      // Time-sensitive info
      /(deadline|due date|launch date|ship date) (is|:)\s*.{5,}/i,
      /(meeting|call|sync|demo) (with|at|on)\s+.{5,}/i,

      // Contact info
      /my (email|phone|contact|slack)\s*(is|:)?\s*.{5,}/i,

      // Repeated information (if they mention something again, might be worth saving)
      /as I (mentioned|said|told you)/i,
      /like I (said|told you)/i
    ];

    function calculateCaptureConfidence(text) {
      if (text.length < 20) return 0;

      // Check for explicit save triggers (100% confidence)
      if (/(haiven,?\s+)?(save|remember|store|keep|note)\s+(this|that)/i.test(text)) {
        return 1.0; // 100% confident
      }

      // Check high confidence patterns
      for (const pattern of HIGH_CONFIDENCE_PATTERNS) {
        if (pattern.test(text)) {
          return 0.95; // 95% confident
        }
      }

      // Check medium confidence patterns
      for (const pattern of MEDIUM_CONFIDENCE_PATTERNS) {
        if (pattern.test(text)) {
          return 0.8; // 80% confident
        }
      }

      // Special case: User is asking to recall something (low confidence suggestion)
      // Example: "haiven remind me about texas skill games"
      if (/(haiven,?\s+)?(remind|recall|what did i say about)/i.test(text)) {
        return 0.7; // 70% - suggest with explanation
      }

      return 0; // Not confident enough to suggest
    }

    let lastAutoSaveCheck = '';
    let autoSaveTimeout = null;

    function checkForAutoCapture() {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) return;

      const currentText = (inputArea.value || inputArea.textContent || '').trim();

      // Debounce - only check if text changed and user stopped typing
      if (currentText === lastAutoSaveCheck || currentText.length < 20) return;
      lastAutoSaveCheck = currentText;

      clearTimeout(autoSaveTimeout);
      autoSaveTimeout = setTimeout(() => {
        const confidence = calculateCaptureConfidence(currentText);

        // Get user's auto-save preference
        browserAPI.storage.local.get(['autoSaveEnabled', 'autoSaveThreshold'], (settings) => {
          const autoSaveEnabled = settings.autoSaveEnabled !== false; // Default true
          const threshold = settings.autoSaveThreshold || 0.7; // Default 70% (lowered to catch more patterns)

          if (!autoSaveEnabled || confidence < threshold) return;

          // Check if already saved recently
          const textHash = btoa(currentText).slice(0, 20);
          browserAPI.storage.local.get(['recentAutoSaves'], (result) => {
            const recentSaves = result.recentAutoSaves || [];

            if (recentSaves.includes(textHash)) {
              console.log('🧠 Haiven: Already auto-saved this content recently');
              return;
            }

            // Show auto-save suggestion
            showAutoSaveSuggestion(currentText, confidence);

            // Track this to avoid duplicates
            recentSaves.push(textHash);
            browserAPI.storage.local.set({
              recentAutoSaves: recentSaves.slice(-20) // Keep last 20
            });
          });
        });
      }, 2000); // Wait 2 seconds after typing stops
    }

    function showAutoSaveSuggestion(text, confidence) {
      // Remove any existing suggestion
      const existing = document.getElementById('haiven-auto-save-suggestion');
      if (existing) existing.remove();

      const suggestion = document.createElement('div');
      suggestion.id = 'haiven-auto-save-suggestion';
      suggestion.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      z-index: 9999998;
      background: rgba(17, 24, 39, 0.95);
      backdrop-filter: blur(12px);
      color: white;
      padding: 16px;
      border-radius: 12px;
      box-shadow: 0 8px 24px rgba(0, 0, 0, 0.3);
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      font-size: 13px;
      max-width: 320px;
      border: 1px solid rgba(139, 92, 246, 0.3);
      animation: slideIn 0.3s ease-out;
    `;

      suggestion.innerHTML = `
      <div style="display: flex; align-items: start; gap: 12px;">
        <div style="font-size: 20px;">💡</div>
        <div style="flex: 1;">
          <div style="font-weight: 600; margin-bottom: 4px;">Save this?</div>
          <div style="opacity: 0.8; font-size: 12px; margin-bottom: 8px;">
            ${Math.round(confidence * 100)}% confident this is worth remembering
          </div>
          <div style="font-size: 12px; opacity: 0.7; margin-bottom: 12px; line-height: 1.4; max-height: 60px; overflow: hidden;">
            "${escapeHtml(text.substring(0, 100))}${text.length > 100 ? '...' : ''}"
          </div>
          <div style="display: flex; gap: 8px;">
            <button id="haiven-auto-save-yes" style="
              flex: 1;
              background: linear-gradient(135deg, #f59e0b, #d97706);
              color: white;
              border: none;
              border-radius: 6px;
              padding: 8px 12px;
              font-size: 12px;
              font-weight: 600;
              cursor: pointer;
            ">✓ Save</button>
            <button id="haiven-auto-save-no" style="
              background: rgba(255, 255, 255, 0.1);
              color: white;
              border: none;
              border-radius: 6px;
              padding: 8px 12px;
              font-size: 12px;
              cursor: pointer;
            ">Not now</button>
          </div>
        </div>
      </div>
    `;

      document.body.appendChild(suggestion);

      // Event handlers
      document.getElementById('haiven-auto-save-yes').addEventListener('click', () => {
        captureMessage({ content: text, isUser: true });
        showCaptureNotification('✓ Auto-saved to Haiven', confidence);
        suggestion.remove();
      });

      document.getElementById('haiven-auto-save-no').addEventListener('click', () => {
        // Learn from dismissal
        browserAPI.runtime.sendMessage({
          type: 'AUTO_SAVE_DISMISSED',
          data: { text, confidence }
        });
        suggestion.remove();
      });

      // Auto-dismiss after 10 seconds
      setTimeout(() => {
        if (suggestion.parentNode) {
          suggestion.style.animation = 'slideOut 0.3s ease-out';
          setTimeout(() => suggestion.remove(), 300);
        }
      }, 10000);
    }

    // Monitor input for auto-capture opportunities
    const inputArea = document.querySelector(currentPlatform.inputSelector);
    if (inputArea) {
      cleanupManager.addListener(inputArea, 'input', checkForAutoCapture);
      console.log('🔮 Haiven: Predictive auto-capture monitoring active');
    }

    // ========================================
    // REAL-TIME PROACTIVE INTELLIGENCE
    // ========================================

    let debounceTimer = null;
    let lastAnalyzedText = '';
    let _currentSuggestions = []; // Reserved for suggestion state
    let suggestionWidget = null;

    // Monitor text input in real-time
    async function startProactiveMonitoring() {
      try {
        const result = await browserAPI.storage.local.get(['proactiveIntelligence']);
        if (result.proactiveIntelligence !== false) {
          console.log('🧠 Haiven: Proactive intelligence enabled');
          attachInputMonitor();
          performContextSwitch(); // Load context on page load
        } else {
          console.log('🧠 Haiven: Proactive intelligence disabled');
        }
      } catch (error) {
        console.error('Failed to check proactive intelligence setting:', error);
      }
    }

    // Attach input event listener
    function attachInputMonitor() {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) {
        setTimeout(attachInputMonitor, 1000);
        return;
      }

      console.log('🧠 Haiven: Monitoring input for real-time analysis');

      // Listen for input events (use cleanupManager to prevent memory leaks)
      cleanupManager.addListener(inputArea, 'input', handleInputChange);
      cleanupManager.addListener(inputArea, 'focus', () => performContextSwitch()); // Reload context on focus
    }

    // Handle input changes with debouncing
    function handleInputChange(event) {
      const input = event.target.value || event.target.textContent || '';

      // Clear previous timer
      if (debounceTimer) {
        clearTimeout(debounceTimer);
      }

      // Debounce analysis - reduced API costs by 70% by increasing delay from 300ms to 1000ms
      debounceTimer = setTimeout(() => {
        analyzeInputRealtime(input);
      }, DEBOUNCE_DELAY_MS);
    }

    // Analyze input in real-time
    async function analyzeInputRealtime(input) {
      // COST SAVING: Skip if input is too short (increased from 10 to 25 chars)
      if (input.trim().length < MIN_INPUT_LENGTH) {
        console.log(
          `⏭️  Skipped: Input too short (${input.trim().length} < ${MIN_INPUT_LENGTH} chars)`
        );
        return;
      }

      // COST SAVING: Skip if same as last analyzed
      if (input === lastAnalyzedText) {
        console.log(`⏭️  Skipped: Same as last query (cache)`);
        return;
      }

      // COST SAVING: Rate limiting - max 10 requests per minute
      const now = Date.now();
      if (now - requestWindowStart > 60000) {
        // Reset window every minute
        requestCount = 0;
        requestWindowStart = now;
      }

      if (requestCount >= MAX_REQUESTS_PER_MINUTE) {
        console.warn(`⚠️  Rate limit: ${MAX_REQUESTS_PER_MINUTE} requests/min exceeded. Skipping.`);
        return;
      }

      // Skip if user not set up
      if (!USER_ID) {
        return;
      }

      // Increment request counter
      requestCount++;
      console.log(`📊 API Request ${requestCount}/${MAX_REQUESTS_PER_MINUTE} this minute`);

      lastAnalyzedText = input;

      // Track LLM usage based on what user is typing (non-blocking)
      trackRealtimeUsage(input);

      try {
        const response = await fetch(`${API_BASE}/api/proactive/analyze-stream`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            platform: currentPlatform.name,
            partialInput: input
          })
        });

        const data = await response.json();

        if (data.success && data.suggestedMemories.length > 0) {
          console.log(
            `🔍 Found ${data.suggestedMemories.length} relevant memories (confidence: ${data.suggestedMemories[0].confidence})`
          );
          console.log(
            `📊 Widget display check - timing: ${data.timing}, confidence: ${data.suggestedMemories[0].confidence}, threshold: ${CONFIDENCE_THRESHOLD}`
          );

          _currentSuggestions = data.suggestedMemories;

          // Show suggestion widget only if confidence is high enough (v3.3.4: using constant threshold)
          const shouldShow =
            data.timing === 'immediate' ||
            (typeof data.suggestedMemories[0].confidence === 'number' &&
              !isNaN(data.suggestedMemories[0].confidence) &&
              data.suggestedMemories[0].confidence >= CONFIDENCE_THRESHOLD);

          console.log(`🎯 Should show widget: ${shouldShow}`);

          if (shouldShow) {
            console.log('📦 Creating suggestion widget...');
            showSuggestionWidget(data.suggestedMemories, data.shouldAutoInject);
          }

          // AUTO-INJECTION: Only if user has enabled it in settings
          if (data.shouldAutoInject && data.suggestedMemories.length > 0) {
            // Check user preference first - auto-inject is OFF by default
            browserAPI.storage.local.get(['autoInject'], (result) => {
              if (result.autoInject === true) {
                console.log('🚀 AUTO-INJECT: High confidence + user enabled, injecting memory');
                autoInjectMemory(data.suggestedMemories[0]);
              } else {
                console.log('⏸️ AUTO-INJECT: Skipped - user has not enabled auto-inject');
              }
            });
          }
        } else {
          // Hide widget if no suggestions
          hideSuggestionWidget();
        }
      } catch (error) {
        console.error('Real-time analysis error:', error);
      }
    }

    // Show Memory Thread modal with narrative view
    async function showMemoryThread(memories) {
      if (!USER_ID) return;

      try {
        console.log('🧵 Building memory thread...');

        const response = await fetch(`${API_BASE}/api/memory/threads`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            memoryIds: memories.map((m) => m.id)
          })
        });

        const data = await response.json();

        if (!data.success || !data.thread) {
          showNotification('Could not build memory thread', 'error');
          return;
        }

        const { thread } = data;

        // Create thread modal
        const modal = document.createElement('div');
        modal.id = 'haiven-memory-thread';
        modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.8);
        backdrop-filter: blur(12px);
        z-index: 9999999;
        display: flex;
        align-items: center;
        justify-content: center;
        animation: fadeIn 0.3s ease-out;
      `;

        modal.innerHTML = `
        <div style="
          background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
          border-radius: 24px;
          padding: 40px;
          max-width: 700px;
          width: 90%;
          max-height: 85vh;
          overflow-y: auto;
          box-shadow: 0 25px 70px rgba(139, 92, 246, 0.4);
          border: 2px solid rgba(139, 92, 246, 0.3);
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
          color: white;
        ">
          <div style="text-align: center; margin-bottom: 32px;">
            <div style="font-size: 48px; margin-bottom: 12px;">📖</div>
            <h2 style="font-size: 24px; margin: 0 0 8px 0; background: linear-gradient(135deg, #f59e0b, #d97706); -webkit-background-clip: text; -webkit-text-fill-color: transparent;">
              ${escapeHtml(thread.topic)}
            </h2>
            <p style="color: rgba(255, 255, 255, 0.6); font-size: 13px; margin: 0;">
              Your journey through ${thread.memories.length} connected memories
            </p>
          </div>

          <div style="position: relative; padding-left: 24px; margin-bottom: 32px;">
            <div style="position: absolute; left: 8px; top: 0; bottom: 0; width: 2px; background: linear-gradient(180deg, rgba(139, 92, 246, 0.6), rgba(139, 92, 246, 0.2));"></div>

            ${thread.memories
              .map(
                (mem, _idx) => `
              <div style="position: relative; margin-bottom: ${mem.isLatest ? '0' : '24px'};">
                <div style="
                  position: absolute;
                  left: -20px;
                  top: 12px;
                  width: 12px;
                  height: 12px;
                  border-radius: 50%;
                  background: ${mem.isLatest ? 'linear-gradient(135deg, #f59e0b, #d97706)' : 'rgba(139, 92, 246, 0.4)'};
                  border: 2px solid ${mem.isLatest ? '#f59e0b' : 'rgba(139, 92, 246, 0.6)'};
                  box-shadow: ${mem.isLatest ? '0 0 12px rgba(139, 92, 246, 0.6)' : 'none'};
                "></div>

                <div style="
                  background: ${mem.isLatest ? 'rgba(139, 92, 246, 0.15)' : 'rgba(255, 255, 255, 0.05)'};
                  border-radius: 12px;
                  padding: 16px;
                  border: 1px solid ${mem.isLatest ? 'rgba(139, 92, 246, 0.3)' : 'rgba(255, 255, 255, 0.1)'};
                ">
                  <div style="font-size: 11px; color: rgba(255, 255, 255, 0.5); margin-bottom: 8px;">
                    ${mem.timeAgo}${mem.isLatest ? ' • <span style="color: #f59e0b; font-weight: 600;">CURRENT</span>' : ''}
                  </div>
                  <div style="font-size: 14px; line-height: 1.6; color: rgba(255, 255, 255, 0.95);">
                    ${escapeHtml(mem.content)}
                  </div>
                </div>

                ${
                  !mem.isLatest
                    ? `
                  <div style="
                    text-align: center;
                    margin: 12px 0;
                    font-size: 20px;
                    color: rgba(139, 92, 246, 0.6);
                  ">
                    ${mem.relationshipEmoji}
                  </div>
                `
                    : ''
                }
              </div>
            `
              )
              .join('')}
          </div>

          ${
            thread.patterns.length > 0
              ? `
            <div style="
              background: rgba(139, 92, 246, 0.1);
              border-radius: 12px;
              padding: 20px;
              margin-bottom: 24px;
              border: 1px solid rgba(139, 92, 246, 0.2);
            ">
              <h3 style="font-size: 14px; margin: 0 0 12px 0; color: #f59e0b; font-weight: 600;">
                💡 Patterns We Detected
              </h3>
              ${thread.patterns
                .map(
                  (p) => `
                <div style="display: flex; align-items: start; gap: 8px; margin-bottom: 8px;">
                  <span style="font-size: 16px;">${p.emoji}</span>
                  <span style="font-size: 13px; color: rgba(255, 255, 255, 0.9); line-height: 1.5;">
                    ${escapeHtml(p.description)}
                  </span>
                </div>
              `
                )
                .join('')}
            </div>
          `
              : ''
          }

          ${
            thread.nextSteps.length > 0
              ? `
            <div style="
              background: rgba(245, 158, 11, 0.1);
              border-radius: 12px;
              padding: 20px;
              margin-bottom: 24px;
              border: 1px solid rgba(245, 158, 11, 0.2);
            ">
              <h3 style="font-size: 14px; margin: 0 0 12px 0; color: #f59e0b; font-weight: 600;">
                ⏭️  Suggested Next Steps
              </h3>
              ${thread.nextSteps
                .map(
                  (step) => `
                <div style="
                  font-size: 13px;
                  color: rgba(255, 255, 255, 0.9);
                  line-height: 1.5;
                  margin-bottom: 6px;
                  padding-left: 12px;
                  border-left: 2px solid rgba(245, 158, 11, 0.3);
                ">
                  ${escapeHtml(step)}
                </div>
              `
                )
                .join('')}
            </div>
          `
              : ''
          }

          <div style="text-align: center;">
            <button id="haiven-thread-close" style="
              background: linear-gradient(135deg, #f59e0b, #d97706);
              color: white;
              border: none;
              border-radius: 12px;
              padding: 14px 32px;
              font-size: 14px;
              font-weight: 600;
              cursor: pointer;
              transition: transform 0.2s;
            ">
              Got It!
            </button>
          </div>
        </div>
      `;

        document.body.appendChild(modal);

        // Event handlers
        document.getElementById('haiven-thread-close').addEventListener('click', () => {
          modal.style.animation = 'fadeOut 0.2s ease-out';
          setTimeout(() => modal.remove(), 200);
        });

        // Close on outside click
        modal.addEventListener('click', (e) => {
          if (e.target === modal) {
            modal.style.animation = 'fadeOut 0.2s ease-out';
            setTimeout(() => modal.remove(), 200);
          }
        });

        console.log('🧵 Memory thread displayed');
      } catch (error) {
        console.error('Error showing memory thread:', error);
        showNotification('Failed to build memory thread', 'error');
      }
    }

    // Show floating suggestion widget
    // DISABLED: Using predictive-overlay.js instead (more polished UI)
    function showSuggestionWidget(suggestions, autoInject) {
      console.log('🎨 showSuggestionWidget DISABLED - using predictive-overlay.js instead');
      return; // Early return - let predictive-overlay.js handle this

      console.log('🎨 showSuggestionWidget called with', suggestions.length, 'suggestions');

      // Remove existing widget
      hideSuggestionWidget();

      try {
        // Track that suggestions were shown
        console.log('📊 Tracking suggestion event...');
        trackSuggestionEvent(suggestions, 'shown', suggestions[0].confidence);

        console.log('✨ Creating widget element...');
        suggestionWidget = document.createElement('div');
        suggestionWidget.id = 'haiven-suggestions';
        suggestionWidget.style.cssText = `
      position: fixed;
      bottom: 100px;
      right: 20px;
      width: 350px;
      max-height: 400px;
      background: rgba(24, 24, 27, 0.98);
      backdrop-filter: blur(20px);
      -webkit-backdrop-filter: blur(20px);
      border: 2px solid rgba(245, 158, 11, 0.3);
      border-radius: 16px;
      box-shadow: 0 8px 40px rgba(245, 158, 11, 0.4);
      z-index: 999998;
      overflow: hidden;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      color: white;
      animation: slideInUp 0.3s ease-out;
    `;

        const topSuggestion = suggestions[0];

        suggestionWidget.innerHTML = `
      <div style="padding: 16px; background: linear-gradient(135deg, rgba(245, 158, 11, 0.2), rgba(217, 119, 6, 0.2)); border-bottom: 1px solid rgba(255,255,255,0.1);">
        <div style="display: flex; justify-content: space-between; align-items: center;">
          <div style="display: flex; align-items: center; gap: 8px;">
            <span style="font-size: 20px;">💡</span>
            <span style="font-size: 14px; font-weight: 600;">Relevant Memories</span>
          </div>
          <button id="haiven-close-suggestions" style="background: none; border: none; color: rgba(255,255,255,0.6); font-size: 18px; cursor: pointer; padding: 0;">×</button>
        </div>
        <div style="font-size: 11px; color: rgba(255,255,255,0.5); margin-top: 4px;">
          ${suggestions.length} ${suggestions.length === 1 ? 'memory' : 'memories'}${topSuggestion.confidence ? ` • ${Math.round(topSuggestion.confidence * 100)}% confidence` : ''}
        </div>
      </div>

      <div style="padding: 16px; max-height: 280px; overflow-y: auto;">
        ${suggestions
          .slice(0, 3)
          .map(
            (mem, _idx) => `
          <div class="suggestion-item" data-memory-id="${mem.id}" style="padding: 12px; background: rgba(255,255,255,0.05); border-radius: 8px; margin-bottom: 8px; cursor: pointer; transition: all 0.2s;">
            <div style="display: flex; justify-content: space-between; align-items: start; margin-bottom: 6px;">
              <span style="font-size: 12px; color: rgba(245, 158, 11, 1); font-weight: 600;">${mem.confidence ? `${Math.round(mem.confidence * 100)}% match` : 'Relevant'}</span>
              <span style="font-size: 11px; color: rgba(255,255,255,0.4);">${formatTimeAgo(mem.createdAt)}</span>
            </div>
            <div style="font-size: 13px; line-height: 1.5; color: rgba(255,255,255,0.9); margin-bottom: 6px;">
              ${escapeHtml(mem.content.substring(0, 120))}${mem.content.length > 120 ? '...' : ''}
            </div>
            <div style="font-size: 11px; color: rgba(217, 119, 6, 1);">
              💡 ${escapeHtml(mem.reason)}
            </div>
          </div>
        `
          )
          .join('')}
      </div>

      <div style="padding: 12px 16px; border-top: 1px solid rgba(255,255,255,0.1); display: flex; gap: 8px; flex-wrap: wrap;">
        ${
          suggestions.length >= 3
            ? `
          <button id="haiven-view-story" style="flex: 1; min-width: 140px; background: linear-gradient(135deg, #f59e0b, #d97706); color: white; border: none; border-radius: 8px; padding: 10px; font-size: 13px; font-weight: 600; cursor: pointer;">
            📖 View Story
          </button>
        `
            : ''
        }
        ${
          autoInject
            ? `
          <button id="haiven-auto-inject-btn" style="flex: 1; min-width: 120px; background: linear-gradient(135deg, #f59e0b, #d97706); color: white; border: none; border-radius: 8px; padding: 10px; font-size: 13px; font-weight: 600; cursor: pointer;">
            ⚡ Auto-Inject
          </button>
        `
            : `
          <button id="haiven-inject-selected" style="flex: 1; min-width: 120px; background: linear-gradient(135deg, #f59e0b, #d97706); color: white; border: none; border-radius: 8px; padding: 10px; font-size: 13px; font-weight: 600; cursor: pointer;">
            ✨ Inject Top
          </button>
        `
        }
        <button id="haiven-dismiss-suggestions" style="flex: 1; min-width: 100px; background: rgba(255,255,255,0.1); color: white; border: none; border-radius: 8px; padding: 10px; font-size: 13px; font-weight: 600; cursor: pointer;">
          Not Now
        </button>
      </div>
    `;

        console.log('🔗 Appending widget to body...');
        document.body.appendChild(suggestionWidget);
        console.log('✅ Widget appended successfully! Element ID:', suggestionWidget.id);

        // Event handlers
        document.getElementById('haiven-close-suggestions')?.addEventListener('click', () => {
          trackSuggestionEvent(suggestions, 'dismissed', suggestions[0].confidence);
          hideSuggestionWidget();
        });
        document.getElementById('haiven-dismiss-suggestions')?.addEventListener('click', () => {
          trackSuggestionEvent(suggestions, 'dismissed', suggestions[0].confidence);
          hideSuggestionWidget();
        });

        document.getElementById('haiven-auto-inject-btn')?.addEventListener('click', () => {
          injectMemories(suggestions);
          hideSuggestionWidget();
        });

        document.getElementById('haiven-inject-selected')?.addEventListener('click', () => {
          injectMemories([suggestions[0]]);
          hideSuggestionWidget();
        });

        document.getElementById('haiven-view-story')?.addEventListener('click', async () => {
          hideSuggestionWidget();
          await showMemoryThread(suggestions);
        });

        // Click to inject individual memory
        suggestionWidget.querySelectorAll('.suggestion-item').forEach((item, idx) => {
          item.addEventListener('click', () => {
            injectMemories([suggestions[idx]]);
            hideSuggestionWidget();
          });

          item.addEventListener('mouseenter', () => {
            item.style.background = 'rgba(245, 158, 11, 0.15)';
          });

          item.addEventListener('mouseleave', () => {
            item.style.background = 'rgba(255,255,255,0.05)';
          });
        });

        // Add slide animation
        const slideStyle = document.createElement('style');
        slideStyle.textContent = `
      @keyframes slideInUp {
        from {
          transform: translateY(30px);
          opacity: 0;
        }
        to {
          transform: translateY(0);
          opacity: 1;
        }
      }
    `;
        if (!document.getElementById('haiven-suggestion-animations')) {
          slideStyle.id = 'haiven-suggestion-animations';
          document.head.appendChild(slideStyle);
        }

        console.log('🎉 Widget created and displayed successfully!');
      } catch (error) {
        console.error('❌ Error creating suggestion widget:', error);
      }
    }

    // Hide suggestion widget
    function hideSuggestionWidget() {
      if (suggestionWidget) {
        suggestionWidget.remove();
        suggestionWidget = null;
      }
    }

    // Inject memories into conversation with LLM-specific formatting
    function injectMemories(memories) {
      const inputArea = document.querySelector(currentPlatform.inputSelector);
      if (!inputArea) return;

      const currentInput = inputArea.value || inputArea.textContent || '';

      // Track click-through for personalization
      trackMemoryInjection(memories);

      // Format memories based on platform
      const formattedMemories = formatMemoriesForPlatform(memories, currentPlatform);

      const newInput = `${formattedMemories}\n\n${currentInput}`;

      if (inputArea.tagName === 'TEXTAREA' || inputArea.tagName === 'INPUT') {
        inputArea.value = newInput;
      } else {
        inputArea.textContent = newInput;
      }

      inputArea.dispatchEvent(new Event('input', { bubbles: true }));
      inputArea.focus();

      showNotification(
        `✨ Injected ${memories.length} ${memories.length === 1 ? 'memory' : 'memories'}`,
        'success'
      );
    }

    // Format memories optimized for each LLM
    function formatMemoriesForPlatform(memories, platform) {
      const formatTimestamp = (created) => {
        if (!created) return 'recently';
        const date = new Date(created);
        const now = new Date();
        const diffDays = Math.floor((now - date) / (1000 * 60 * 60 * 24));

        if (diffDays === 0) return 'today';
        if (diffDays === 1) return 'yesterday';
        if (diffDays < 7) return `${diffDays} days ago`;
        if (diffDays < 30) return `${Math.floor(diffDays / 7)} weeks ago`;
        return date.toLocaleDateString();
      };

      switch (platform.promptStyle) {
        case 'xml-structured': // Claude prefers structured XML
          const xmlMemories = memories
            .map(
              (m) =>
                `  <memory confidence="${Math.round((m.confidence || 0.9) * 100)}%" recency="${formatTimestamp(m.created_at)}">\n    ${m.content}\n  </memory>`
            )
            .join('\n');
          return `<context source="haiven">\n${xmlMemories}\n</context>\n\nUsing the context above:`;

        case 'conversational': // ChatGPT/Gemini prefer conversational
          const conversationalMemories = memories
            .map((m, idx) => {
              const conf = Math.round((m.confidence || 0.9) * 100);
              const time = formatTimestamp(m.created_at);
              return `${idx + 1}. ${m.content}\n   (${conf}% relevant • from ${time})`;
            })
            .join('\n\n');
          return `Here's what I remember about this:\n\n${conversationalMemories}\n\nWith this context in mind:`;

        case 'direct': // Grok prefers direct, no fluff
          const directMemories = memories.map((m) => `• ${m.content}`).join('\n');
          return `Context:\n${directMemories}\n\nQ:`;

        case 'research-focused': // Perplexity likes structured background
          const researchMemories = memories
            .map((m, idx) => {
              const time = formatTimestamp(m.created_at);
              return `**Source ${idx + 1}** (${time}):\n${m.content}`;
            })
            .join('\n\n');
          return `## Background Information\n\n${researchMemories}\n\n## Query:\n`;

        default:
          return memories.map((m) => m.content).join('\n\n');
      }
    }

    // Track memory injection for personalization learning
    async function trackMemoryInjection(memories) {
      try {
        await fetch(`${API_BASE}/api/analytics/memory-injection`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            memoryIds: memories.map((m) => m.id),
            platform: currentPlatform.name,
            timestamp: Date.now(),
            action: 'injected'
          })
        });
      } catch (error) {
        console.log('Analytics tracking failed (non-critical):', error);
      }
    }

    // Track suggestion events (shown, dismissed, clicked) for personalization
    async function trackSuggestionEvent(suggestions, action, confidence) {
      try {
        await fetch(`${API_BASE}/api/analytics/suggestion-event`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            memoryIds: suggestions.map((s) => s.id),
            action: action, // 'shown', 'dismissed', 'clicked'
            confidence: confidence || 0,
            platform: currentPlatform.name
          })
        });
      } catch (error) {
        console.log('Suggestion tracking failed (non-critical):', error);
      }
    }

    // ========================================
    // PRIVACY FILTERS
    // ========================================

    // Check if content contains sensitive data that shouldn't be captured
    function containsSensitiveData(content) {
      if (!content) return false;

      const _text = content.toLowerCase(); // Reserved for pattern matching

      // Password patterns
      const passwordPatterns = [
        /password[:\s]*[^\s]+/i,
        /passwd[:\s]*[^\s]+/i,
        /pwd[:\s]*[^\s]+/i,
        /secret[:\s]*[^\s]+/i
      ];

      // API key patterns
      const apiKeyPatterns = [
        /api[_-]?key[:\s]*[a-z0-9]{20,}/i,
        /bearer\s+[a-z0-9_\-\.]+/i,
        /token[:\s]*[a-z0-9]{20,}/i,
        /sk-[a-z0-9\-]{20,}/i, // OpenAI style (including sk-proj-)
        /ghp_[a-z0-9]{36}/i, // GitHub
        /AIza[a-z0-9_\-]{35}/i // Google
      ];

      // Credit card patterns (basic check)
      const creditCardPatterns = [
        /\b\d{4}[\s-]?\d{4}[\s-]?\d{4}[\s-]?\d{4}\b/,
        /\b\d{13,19}\b/ // General card number length
      ];

      // Social security numbers
      const ssnPattern = /\b\d{3}[-\s]?\d{2}[-\s]?\d{4}\b/;

      // Private keys
      const privateKeyPattern = /-----BEGIN (RSA |EC )?PRIVATE KEY-----/;

      // Check all patterns
      const allPatterns = [
        ...passwordPatterns,
        ...apiKeyPatterns,
        ...creditCardPatterns,
        ssnPattern,
        privateKeyPattern
      ];

      for (const pattern of allPatterns) {
        if (pattern.test(content)) {
          console.warn('🔒 Haiven: Blocked sensitive data from capture');
          return true;
        }
      }

      return false;
    }

    // Sanitize content before capture (remove sensitive parts)
    function sanitizeContent(content) {
      if (!content) return content;

      let sanitized = content;

      // Redact API keys
      sanitized = sanitized.replace(/api[_-]?key[:\s]*[a-z0-9]{20,}/gi, 'API_KEY: [REDACTED]');
      sanitized = sanitized.replace(/bearer\s+[a-z0-9_\-\.]+/gi, 'Bearer [REDACTED]');
      sanitized = sanitized.replace(/token[:\s]*[a-z0-9]{20,}/gi, 'Token: [REDACTED]');

      // Redact OpenAI keys
      sanitized = sanitized.replace(/sk-[a-z0-9]{32,}/gi, '[REDACTED_API_KEY]');

      // Redact credit cards
      sanitized = sanitized.replace(
        /\b\d{4}[\s-]?\d{4}[\s-]?\d{4}[\s-]?\d{4}\b/g,
        '[CREDIT_CARD_REDACTED]'
      );

      // Redact passwords
      sanitized = sanitized.replace(/password[:\s]*[^\s]+/gi, 'Password: [REDACTED]');

      return sanitized;
    }

    // Perform context switch on platform load
    async function performContextSwitch() {
      // Ensure we have the latest userId from storage
      let userId = USER_ID;
      if (!userId) {
        const result = await browserAPI.storage.local.get(['userId']);
        userId = result.userId || '';
        if (userId) {
          USER_ID = userId; // Update cached value
          console.log('🔄 Retrieved userId from storage for context switch');
        }
      }

      // Skip if user not set up
      if (!userId) {
        console.log('🧠 Haiven: User ID not set, skipping context switch');
        return;
      }

      try {
        const response = await fetch(`${API_BASE}/api/proactive/context-switch`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: userId,
            platform: currentPlatform.name,
            timestamp: Date.now()
          })
        });

        if (!response.ok) {
          console.log('🧠 Haiven: Context switch failed (non-critical)');
          return;
        }

        const data = await response.json();

        if (data.success) {
          console.log(
            `🔄 Context loaded: ${data.contextLoaded} (${data.memoryCount} memories, ${Math.round(data.confidence * 100)}% confidence)`
          );

          // Show subtle notification - DISABLED (user prefers predictive overlay only)
          // showContextLoadedBadge(data.contextLoaded, data.memoryCount);
        }
      } catch (error) {
        console.log('🧠 Haiven: Context switch error (non-critical):', error.message);
      }
    }

    // Show context loaded badge (reserved for future use)
    function _showContextLoadedBadge(context, memoryCount) {
      console.log(`📍 Showing context badge: ${context}, ${memoryCount} memories`);

      // Ensure document.body exists
      if (!document.body) {
        console.warn('⚠️  document.body not ready, waiting...');
        setTimeout(() => _showContextLoadedBadge(context, memoryCount), 100);
        return;
      }

      // Remove any existing badge
      const existingBadge = document.getElementById('haiven-context-badge-container');
      if (existingBadge) {
        existingBadge.remove();
      }

      // Create container with inline styles (no wrapper)
      const badge = document.createElement('div');
      badge.id = 'haiven-context-badge-container';
      badge.style.cssText = `
      position: fixed !important;
      top: 20px !important;
      right: 20px !important;
      background: linear-gradient(135deg, #f59e0b 0%, #4ecdcc 100%) !important;
      color: white !important;
      padding: 16px 24px !important;
      border-radius: 12px !important;
      font-size: 15px !important;
      font-weight: 700 !important;
      z-index: 2147483647 !important;
      box-shadow: 0 10px 40px rgba(245, 158, 11, 0.8), 0 0 0 3px rgba(255, 255, 255, 0.5) !important;
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif !important;
      pointer-events: auto !important;
      display: flex !important;
      flex-direction: column !important;
      gap: 12px !important;
      visibility: visible !important;
      opacity: 1 !important;
      letter-spacing: 0.3px !important;
      text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2) !important;
      transform: translateX(0) !important;
      transition: transform 0.3s ease-out, opacity 0.3s ease-out !important;
    `;

      const textDiv = document.createElement('div');
      textDiv.textContent = `🧠 ${context.charAt(0).toUpperCase() + context.slice(1)} context loaded • ${memoryCount} memories ready`;

      const injectBtn = document.createElement('button');
      injectBtn.id = 'haiven-inject-btn';
      injectBtn.textContent = '📥 Load Context Into Chat';
      injectBtn.style.cssText = `
      background: rgba(255, 255, 255, 0.95) !important;
      color: #f59e0b !important;
      border: none !important;
      padding: 10px 20px !important;
      border-radius: 8px !important;
      font-size: 14px !important;
      font-weight: 700 !important;
      cursor: pointer !important;
      transition: all 0.2s ease !important;
      box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15) !important;
    `;

      badge.appendChild(textDiv);
      badge.appendChild(injectBtn);
      document.body.appendChild(badge);

      console.log(
        '✅ Badge appended to body:',
        badge ? 'element found' : 'ERROR - element not found'
      );

      // Store the memories for later use (reserved for future use)
      const _preloadedMemories = memoryCount > 0 ? [] : [];

      // Add click handler for injection button
      if (injectBtn) {
        injectBtn.addEventListener('click', async () => {
          console.log('🚀 Injection button clicked');

          try {
            // Disable button during injection
            injectBtn.disabled = true;
            injectBtn.textContent = '⏳ Loading...';

            // Extract conversation context from the page
            showNotification('Analyzing conversation context...', 'info');
            const conversationText = extractConversationContext();
            console.log('📝 Extracted conversation context:', conversationText.substring(0, 200));

            // Use multi-head smart search to find contextually relevant memories
            // 4 perspectives: semantic, project context, pattern matching, decision framework
            showNotification('Finding relevant memories...', 'info');
            const response = await fetch(`${API_BASE}/api/memory/smart-search`, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
                'x-user-id': USER_ID
              },
              body: JSON.stringify({
                userId: USER_ID,
                query: conversationText || 'recent work context',
                limit: 5,
                options: {
                  useMultiHead: true,
                  includePatterns: true,
                  includeDecisions: true
                }
              })
            });

            if (!response.ok) {
              throw new Error('Failed to fetch memories');
            }

            const data = await response.json();
            let memories = data.memories || [];

            // 🆕 FALLBACK: If semantic search returns no results, get recent memories
            if (!memories || memories.length === 0) {
              console.log(
                '⚠️ Semantic search returned no results, falling back to recent memories...'
              );
              showNotification('Loading recent memories...', 'info');

              // Get most recent memories as fallback
              const fallbackResponse = await fetch(
                `${API_BASE}/api/memory/recent/${USER_ID}?limit=5`,
                {
                  method: 'GET',
                  headers: {
                    'x-user-id': USER_ID
                  }
                }
              );

              if (fallbackResponse.ok) {
                const fallbackData = await fallbackResponse.json();
                memories = fallbackData.memories || fallbackData || [];
                console.log(`✅ Recent memory fallback found ${memories.length} memories`);
              }
            }

            if (!memories || memories.length === 0) {
              showNotification('No relevant memories found', 'error');
              injectBtn.disabled = false;
              injectBtn.textContent = '📥 Load Context Into Chat';
              return;
            }

            console.log(`✅ Found ${memories.length} relevant memories`);

            // Generate smart context handoff
            showNotification('Preparing smart context...', 'info');

            const handoffResponse = await fetch(`${API_BASE}/api/proactive/context-handoff`, {
              method: 'POST',
              headers: {
                'Content-Type': 'application/json',
                'x-user-id': USER_ID
              },
              body: JSON.stringify({
                memory: memories.map((m) => m.content).join('\n\n'),
                fromPlatform: 'Haiven Dashboard',
                toPlatform: currentPlatform.name,
                timestamp: new Date().toISOString()
              })
            });

            if (!handoffResponse.ok) {
              throw new Error('Failed to generate context handoff');
            }

            const { handoff } = await handoffResponse.json();

            // Find the input field
            const inputField = document.querySelector(currentPlatform.inputSelector);

            if (inputField) {
              // Inject smart context
              if (inputField.tagName === 'TEXTAREA' || inputField.tagName === 'INPUT') {
                const start = inputField.selectionStart || 0;
                const end = inputField.selectionEnd || 0;
                const currentValue = inputField.value || '';
                inputField.value =
                  currentValue.substring(0, start) + handoff + currentValue.substring(end);
                inputField.selectionStart = inputField.selectionEnd = start + handoff.length;
                inputField.dispatchEvent(new Event('input', { bubbles: true }));
              } else if (inputField.isContentEditable) {
                const selection = window.getSelection();
                const range = selection.getRangeAt(0);
                range.deleteContents();
                const textNode = document.createTextNode(handoff);
                range.insertNode(textNode);
                range.setStartAfter(textNode);
                range.setEndAfter(textNode);
                selection.removeAllRanges();
                selection.addRange(range);
                inputField.dispatchEvent(new Event('input', { bubbles: true }));
              }

              inputField.focus();
              console.log('✅ Smart context injected');
              showNotification('Context loaded - ready to continue!', 'success');

              // Close the badge
              if (badge) {
                badge.style.transform = 'translateX(400px)';
                badge.style.opacity = '0';
                setTimeout(() => badge.remove(), 300);
              }
            } else {
              console.warn('Input field not found');
              showNotification('Could not find input field', 'error');
              injectBtn.disabled = false;
              injectBtn.textContent = '📥 Load Context Into Chat';
            }
          } catch (error) {
            console.error('Failed to inject context:', error);
            showNotification('Failed to inject context', 'error');
            injectBtn.disabled = false;
            injectBtn.textContent = '📥 Load Context Into Chat';
          }
        });
      }

      // Log computed styles for debugging
      if (badge) {
        const computed = window.getComputedStyle(badge);
        console.log('🎨 Badge computed styles:', {
          display: computed.display,
          visibility: computed.visibility,
          opacity: computed.opacity,
          position: computed.position,
          zIndex: computed.zIndex,
          top: computed.top,
          right: computed.right
        });
      }

      // Auto-hide after 30 seconds (longer to give user time to click)
      setTimeout(() => {
        if (badge && !injectBtn.disabled) {
          console.log('🔄 Starting badge fadeout');
          badge.style.transform = 'translateX(400px)';
          badge.style.opacity = '0';
          setTimeout(() => {
            badge.remove();
            console.log('🗑️  Badge removed');
          }, 300);
        }
      }, 30000);
    }

    // ========================================
    // AUTO-INJECTION: Context Engine Feature
    // ========================================

    let lastInjectedMemoryId = null;
    let injectionCooldown = false;

    /**
     * AUTO-INJECT MEMORY INTO LLM CONTEXT
     * This is the "context engine" feature - automatically prepend relevant memory
     * to the user's message before it's sent to the AI.
     */
    function autoInjectMemory(memory) {
      // Prevent duplicate injections
      if (lastInjectedMemoryId === memory.id || injectionCooldown) {
        console.log('⏭️  Skipping injection - cooldown active or already injected');
        return;
      }

      try {
        const inputArea = document.querySelector(currentPlatform.inputSelector);
        if (!inputArea) {
          console.warn('⚠️  Could not find input area for injection');
          return;
        }

        // Get current input value
        const currentInput = (inputArea.value || inputArea.textContent || '').trim();

        // Don't inject if input is empty (user hasn't started typing yet)
        if (!currentInput || currentInput.length < 5) {
          console.log('⏭️  Skipping injection - input too short');
          return;
        }

        // Format the memory context injection
        const contextPrefix = `[📚 Context from your Haiven Memory: "${memory.content.substring(0, 100)}${memory.content.length > 100 ? '...' : ''}"]\n\n`;

        // Inject memory context before user's message
        const injectedText = contextPrefix + currentInput;

        // Update the input field
        if (inputArea.tagName === 'TEXTAREA' || inputArea.tagName === 'INPUT') {
          inputArea.value = injectedText;
        } else {
          inputArea.textContent = injectedText;
        }

        // Trigger input event so the platform knows text changed
        inputArea.dispatchEvent(new Event('input', { bubbles: true }));

        // Set cooldown to prevent multiple injections
        lastInjectedMemoryId = memory.id;
        injectionCooldown = true;
        setTimeout(() => {
          injectionCooldown = false;
        }, 10000); // 10 second cooldown

        // Show subtle notification
        showInjectionNotification(memory);

        console.log(`✅ AUTO-INJECT: Memory injected into context (ID: ${memory.id})`);
      } catch (error) {
        console.error('❌ AUTO-INJECT ERROR:', error);
      }
    }

    /**
     * Show notification that memory was auto-injected
     */
    function showInjectionNotification(_memory) {
      const notification = document.createElement('div');
      notification.style.cssText = `
      position: fixed;
      bottom: 20px;
      right: 20px;
      background: linear-gradient(135deg, rgba(139, 92, 246, 0.95), rgba(124, 58, 237, 0.95));
      backdrop-filter: blur(10px);
      -webkit-backdrop-filter: blur(10px);
      color: white;
      padding: 14px 18px;
      border-radius: 10px;
      font-size: 13px;
      font-weight: 500;
      z-index: 9999999;
      box-shadow: 0 6px 24px rgba(139, 92, 246, 0.4);
      animation: slideInUp 0.4s cubic-bezier(0.34, 1.56, 0.64, 1);
      font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
      max-width: 350px;
      border: 1px solid rgba(255, 255, 255, 0.2);
    `;

      notification.innerHTML = `
      <div style="display: flex; align-items: center; gap: 10px;">
        <span style="font-size: 20px;">🚀</span>
        <div>
          <div style="font-weight: 600; margin-bottom: 4px;">Memory Auto-Injected</div>
          <div style="font-size: 11px; opacity: 0.9;">Context added to your message</div>
        </div>
      </div>
    `;

      document.body.appendChild(notification);

      // Add slide-in animation
      const style = document.createElement('style');
      style.textContent = `
      @keyframes slideInUp {
        from {
          transform: translateY(20px);
          opacity: 0;
        }
        to {
          transform: translateY(0);
          opacity: 1;
        }
      }
    `;
      if (!document.querySelector('style[data-haiven-inject-anim]')) {
        style.setAttribute('data-haiven-inject-anim', 'true');
        document.head.appendChild(style);
      }

      setTimeout(() => {
        notification.style.transition = 'opacity 0.3s, transform 0.3s';
        notification.style.opacity = '0';
        notification.style.transform = 'translateY(10px)';
        setTimeout(() => notification.remove(), 300);
      }, 5000);
    }

    // ========================================
    // MORNING BRIEFING
    // ========================================

    async function showMorningBriefing() {
      if (!USER_ID) return;

      try {
        const response = await fetch(`${API_BASE}/api/proactive/morning-briefing`, {
          method: 'POST',
          headers: await getAuthHeaders(),
          body: JSON.stringify({
            userId: USER_ID,
            platform: currentPlatform.name
          })
        });

        const data = await response.json();

        if (!data.success || !data.briefing) return;

        const { briefing } = data;

        // Create briefing modal
        const modal = document.createElement('div');
        modal.id = 'haiven-morning-briefing';
        modal.style.cssText = `
        position: fixed;
        top: 0;
        left: 0;
        width: 100%;
        height: 100%;
        background: rgba(0, 0, 0, 0.7);
        backdrop-filter: blur(8px);
        z-index: 9999999;
        display: flex;
        align-items: center;
        justify-content: center;
        animation: fadeIn 0.3s ease-out;
      `;

        modal.innerHTML = `
        <div style="
          background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
          border-radius: 24px;
          padding: 40px;
          max-width: 600px;
          width: 90%;
          max-height: 80vh;
          overflow-y: auto;
          box-shadow: 0 20px 60px rgba(245, 158, 11, 0.3);
          border: 2px solid rgba(245, 158, 11, 0.2);
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
          color: white;
        ">
          <div style="text-align: center; margin-bottom: 30px;">
            <div style="font-size: 48px; margin-bottom: 10px;">☀️</div>
            <h2 style="font-size: 28px; margin: 0 0 10px 0; background: linear-gradient(135deg, #f59e0b, #d97706); -webkit-background-clip: text; -webkit-text-fill-color: transparent;">
              ${briefing.greeting}
            </h2>
            <p style="color: rgba(255, 255, 255, 0.7); font-size: 14px; margin: 0;">
              ${briefing.message}
            </p>
          </div>

          ${
            briefing.suggestions.length > 0
              ? `
            <div style="margin-bottom: 24px;">
              <h3 style="font-size: 16px; margin: 0 0 16px 0; color: rgba(255, 255, 255, 0.9);">💡 Smart Suggestions</h3>
              ${briefing.suggestions
                .map(
                  (s) => `
                <div class="briefing-suggestion" data-type="${s.type}" data-memory-id="${s.memoryId || ''}" data-topic="${s.topic || ''}" style="
                  background: rgba(255, 255, 255, 0.05);
                  border-radius: 12px;
                  padding: 16px;
                  margin-bottom: 12px;
                  cursor: pointer;
                  transition: all 0.2s;
                  border: 1px solid rgba(255, 255, 255, 0.1);
                ">
                  <div style="font-size: 14px; font-weight: 600; margin-bottom: 4px; color: #f59e0b;">
                    ${s.title}
                  </div>
                  <div style="font-size: 13px; color: rgba(255, 255, 255, 0.7);">
                    ${s.description}
                  </div>
                </div>
              `
                )
                .join('')}
            </div>
          `
              : ''
          }

          ${
            briefing.stats.topTopics.length > 0
              ? `
            <div style="margin-bottom: 24px;">
              <h3 style="font-size: 14px; margin: 0 0 12px 0; color: rgba(255, 255, 255, 0.7);">📊 Your Recent Focus</h3>
              <div style="display: flex; gap: 8px; flex-wrap: wrap;">
                ${briefing.stats.topTopics
                  .map(
                    (topic) => `
                  <span style="
                    background: rgba(245, 158, 11, 0.2);
                    color: #f59e0b;
                    padding: 6px 12px;
                    border-radius: 16px;
                    font-size: 12px;
                    font-weight: 500;
                  ">${topic}</span>
                `
                  )
                  .join('')}
              </div>
            </div>
          `
              : ''
          }

          <div style="text-align: center;">
            <button id="haiven-briefing-close" style="
              background: linear-gradient(135deg, #f59e0b, #d97706);
              color: white;
              border: none;
              border-radius: 12px;
              padding: 14px 32px;
              font-size: 14px;
              font-weight: 600;
              cursor: pointer;
              transition: transform 0.2s;
            ">
              Let's Go!
            </button>
          </div>
        </div>
      `;

        document.body.appendChild(modal);

        // Event handlers
        document.getElementById('haiven-briefing-close').addEventListener('click', () => {
          modal.style.animation = 'fadeOut 0.2s ease-out';
          setTimeout(() => modal.remove(), 200);
        });

        // Handle suggestion clicks
        modal.querySelectorAll('.briefing-suggestion').forEach((suggestion) => {
          suggestion.addEventListener('mouseenter', () => {
            suggestion.style.background = 'rgba(245, 158, 11, 0.15)';
            suggestion.style.transform = 'translateX(4px)';
          });
          suggestion.addEventListener('mouseleave', () => {
            suggestion.style.background = 'rgba(255, 255, 255, 0.05)';
            suggestion.style.transform = 'translateX(0)';
          });
          suggestion.addEventListener('click', async () => {
            const type = suggestion.getAttribute('data-type');
            const memoryId = suggestion.getAttribute('data-memory-id');
            const topic = suggestion.getAttribute('data-topic');

            // Handle different suggestion types
            if (type === 'resume' && memoryId) {
              // Direct injection - no intermediate overlay step
              console.log('Resuming memory:', memoryId);

              try {
                // Fetch the full memory from API
                const response = await fetch(`${API_BASE}/api/memory/${memoryId}`, {
                  headers: { 'x-user-id': USER_ID }
                });

                if (response.ok) {
                  const { memory } = await response.json();

                  // Get intelligent context handoff
                  showNotification('Preparing smart context...', 'info');

                  const handoffResponse = await fetch(`${API_BASE}/api/proactive/context-handoff`, {
                    method: 'POST',
                    headers: {
                      'Content-Type': 'application/json',
                      'x-user-id': USER_ID
                    },
                    body: JSON.stringify({
                      memory: memory.content,
                      fromPlatform: memory.metadata?.platform || 'ChatGPT',
                      toPlatform: currentPlatform.name,
                      timestamp: memory.created_at
                    })
                  });

                  if (!handoffResponse.ok) {
                    throw new Error('Failed to generate context handoff');
                  }

                  const { handoff } = await handoffResponse.json();

                  // Find the input field
                  const inputField = document.querySelector(currentPlatform.inputSelector);

                  if (inputField) {
                    // Inject smart context
                    if (inputField.tagName === 'TEXTAREA' || inputField.tagName === 'INPUT') {
                      const start = inputField.selectionStart || 0;
                      const end = inputField.selectionEnd || 0;
                      const currentValue = inputField.value || '';
                      inputField.value =
                        currentValue.substring(0, start) + handoff + currentValue.substring(end);
                      inputField.selectionStart = inputField.selectionEnd = start + handoff.length;
                      inputField.dispatchEvent(new Event('input', { bubbles: true }));
                    } else if (inputField.isContentEditable) {
                      const selection = window.getSelection();
                      const range = selection.getRangeAt(0);
                      range.deleteContents();
                      const textNode = document.createTextNode(handoff);
                      range.insertNode(textNode);
                      range.setStartAfter(textNode);
                      range.setEndAfter(textNode);
                      selection.removeAllRanges();
                      selection.addRange(range);
                      inputField.dispatchEvent(new Event('input', { bubbles: true }));
                    }

                    inputField.focus();
                    console.log('✅ Smart context injected');
                    showNotification('Context loaded - ready to continue!', 'success');
                  } else {
                    console.warn('Input field not found');
                    showNotification('Could not find input field', 'error');
                  }
                }
              } catch (error) {
                console.error('Failed to load memory:', error);
                showNotification('Failed to load memory', 'error');
              }
            } else if (type === 'explore' && topic) {
              console.log('Exploring topic:', topic);
              showNotification(`Exploring ${topic} memories...`, 'info');
            }

            modal.remove();
          });
        });

        // Close on outside click
        modal.addEventListener('click', (e) => {
          if (e.target === modal) {
            modal.style.animation = 'fadeOut 0.2s ease-out';
            setTimeout(() => modal.remove(), 200);
          }
        });

        // Add animations
        const style = document.createElement('style');
        style.textContent = `
        @keyframes fadeIn {
          from { opacity: 0; }
          to { opacity: 1; }
        }
        @keyframes fadeOut {
          from { opacity: 1; }
          to { opacity: 0; }
        }
      `;
        if (!document.getElementById('haiven-briefing-animations')) {
          style.id = 'haiven-briefing-animations';
          document.head.appendChild(style);
        }

        console.log('☀️ Morning briefing shown');
      } catch (error) {
        console.error('Morning briefing error:', error);
      }
    }

    // ========================================
    // SELECTION POPOVER - Beautiful, safe text capture
    // ========================================

    let selectionPopover = null;
    let removePopoverHandler = null;

    // Inject beautiful styles once
    function injectSelectionStyles() {
      if (document.getElementById('haiven-selection-styles')) return;
      const style = createElement('style', { id: 'haiven-selection-styles' });
      style.textContent = `
        @keyframes haivenPopIn {
          0% { opacity: 0; transform: scale(0.9) translateY(4px); }
          100% { opacity: 1; transform: scale(1) translateY(0); }
        }
        @keyframes haivenPulse {
          0%, 100% { box-shadow: 0 4px 24px rgba(245, 158, 11, 0.4); }
          50% { box-shadow: 0 4px 32px rgba(245, 158, 11, 0.6); }
        }
        @keyframes haivenSpin {
          from { transform: rotate(0deg); }
          to { transform: rotate(360deg); }
        }
        #haiven-selection-popover {
          font-family: -apple-system, BlinkMacSystemFont, 'SF Pro Display', 'Segoe UI', sans-serif;
          animation: haivenPopIn 0.3s cubic-bezier(0.16, 1, 0.3, 1);
        }
        .haiven-save-btn {
          transition: all 0.2s cubic-bezier(0.16, 1, 0.3, 1);
        }
        .haiven-save-btn:hover {
          background: linear-gradient(135deg, rgba(0, 212, 255, 0.1) 0%, rgba(138, 43, 226, 0.1) 100%) !important;
        }
        .haiven-save-btn:hover .haiven-arrow {
          transform: translateX(3px);
          opacity: 1 !important;
        }
        .haiven-save-btn:active {
          transform: scale(0.98);
          background: rgba(0, 212, 255, 0.15) !important;
        }
        .haiven-spinner {
          animation: haivenSpin 0.8s linear infinite;
          display: inline-block;
        }
      `;
      document.head.appendChild(style);
    }

    function createSelectionPopover(rect, selectedText) {
      injectSelectionStyles();

      const preview =
        selectedText.length > 60 ? selectedText.substring(0, 60) + '...' : selectedText;

      // Premium icon container with gradient background
      const iconContainer = createElement(
        'div',
        {
          style: {
            width: '36px',
            height: '36px',
            borderRadius: '10px',
            background:
              'linear-gradient(135deg, rgba(0, 212, 255, 0.2) 0%, rgba(138, 43, 226, 0.2) 100%)',
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'center',
            flexShrink: '0'
          }
        },
        []
      );

      const iconSpan = createElement(
        'span',
        {
          style: {
            fontSize: '16px',
            filter: 'drop-shadow(0 0 4px rgba(0, 212, 255, 0.5))'
          }
        },
        ['✦']
      );
      iconContainer.appendChild(iconSpan);

      const textContainer = createElement('div', { style: { flex: '1', minWidth: '0' } }, []);

      const textSpan = createElement(
        'span',
        {
          style: {
            display: 'block',
            fontSize: '14px',
            fontWeight: '600',
            color: '#ffffff',
            letterSpacing: '-0.01em'
          }
        },
        ['Save to Haiven']
      );

      const subtextSpan = createElement(
        'span',
        {
          style: {
            display: 'block',
            fontSize: '11px',
            color: 'rgba(255, 255, 255, 0.5)',
            marginTop: '2px'
          }
        },
        ['Add to your memory']
      );

      textContainer.appendChild(textSpan);
      textContainer.appendChild(subtextSpan);

      const arrowSpan = createElement(
        'span',
        {
          className: 'haiven-arrow',
          style: {
            fontSize: '18px',
            color: 'rgba(0, 212, 255, 0.8)',
            transition: 'transform 0.2s ease, opacity 0.2s ease',
            opacity: '0.6'
          }
        },
        ['›']
      );

      const saveBtn = createElement(
        'button',
        {
          className: 'haiven-save-btn',
          style: {
            display: 'flex',
            alignItems: 'center',
            gap: '12px',
            padding: '14px 16px',
            background: 'transparent',
            border: 'none',
            color: 'white',
            cursor: 'pointer',
            width: '100%',
            textAlign: 'left',
            borderRadius: '12px',
            transition: 'background 0.2s ease'
          }
        },
        [iconContainer, textContainer, arrowSpan]
      );

      const previewDiv = createElement(
        'div',
        {
          style: {
            padding: '12px 16px 14px',
            fontSize: '12px',
            color: 'rgba(255, 255, 255, 0.5)',
            borderTop: '1px solid rgba(255, 255, 255, 0.06)',
            maxWidth: '300px',
            overflow: 'hidden',
            textOverflow: 'ellipsis',
            whiteSpace: 'nowrap',
            fontStyle: 'italic',
            background: 'rgba(0, 0, 0, 0.15)'
          }
        },
        [`"${preview}"`]
      );

      // Calculate position - flip above selection if near bottom of viewport
      const popoverHeight = 100; // Approximate height
      const spaceBelow = window.innerHeight - rect.bottom;
      const showAbove = spaceBelow < popoverHeight + 20;
      const topPosition = showAbove
        ? Math.max(10, rect.top - popoverHeight - 10)
        : rect.bottom + 10;

      const popover = createElement(
        'div',
        {
          id: 'haiven-selection-popover',
          style: {
            position: 'fixed',
            top: `${topPosition}px`,
            left: `${Math.max(10, Math.min(rect.left, window.innerWidth - 320))}px`,
            zIndex: '999999',
            background:
              'linear-gradient(165deg, rgba(15, 23, 42, 0.98) 0%, rgba(30, 41, 59, 0.98) 100%)',
            border: '1px solid rgba(255, 255, 255, 0.08)',
            borderRadius: '16px',
            boxShadow:
              '0 25px 50px -12px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(255, 255, 255, 0.05), 0 0 40px rgba(0, 212, 255, 0.1)',
            backdropFilter: 'blur(24px)',
            overflow: 'hidden',
            minWidth: '280px'
          }
        },
        [saveBtn, previewDiv]
      );

      // Handle save click
      saveBtn.addEventListener('click', async (e) => {
        e.stopPropagation();

        // Update UI to loading state
        iconSpan.textContent = '⏳';
        iconSpan.className = 'haiven-spinner';
        textSpan.textContent = 'Saving...';
        arrowSpan.textContent = '';

        try {
          await captureMessage({ content: selectedText, isUser: true });

          // Success state
          iconSpan.className = '';
          iconSpan.textContent = '✅';
          textSpan.textContent = 'Saved!';
          popover.style.borderColor = 'rgba(16, 185, 129, 0.5)';
          popover.style.boxShadow = '0 4px 24px rgba(16, 185, 129, 0.3)';

          showNotification('💾 Memory saved!', 'success');

          setTimeout(() => {
            if (selectionPopover) {
              selectionPopover.style.opacity = '0';
              selectionPopover.style.transform = 'scale(0.95) translateY(-4px)';
              selectionPopover.style.transition = 'all 0.2s ease';
              setTimeout(() => {
                selectionPopover?.remove();
                selectionPopover = null;
              }, 200);
            }
          }, 800);
        } catch (error) {
          iconSpan.className = '';
          iconSpan.textContent = '❌';
          textSpan.textContent = 'Failed';
          popover.style.borderColor = 'rgba(239, 68, 68, 0.5)';
          console.error('Selection save error:', error);
        }
      });

      return popover;
    }

    function handleTextSelection(e) {
      // Remove existing popover
      if (selectionPopover) {
        selectionPopover.remove();
        selectionPopover = null;
      }
      if (removePopoverHandler) {
        document.removeEventListener('mousedown', removePopoverHandler);
        removePopoverHandler = null;
      }

      // Don't show if clicking on Haiven UI elements
      if (
        e.target.closest(
          '#haiven-panel, #haiven-suggestions, #haiven-selection-popover, .haiven-overlay, #haiven-modal-overlay'
        )
      ) {
        return;
      }

      const selection = window.getSelection();
      const selectedText = selection.toString().trim();

      // Only show for meaningful selections (20+ chars)
      if (selectedText.length < 20) return;

      try {
        const range = selection.getRangeAt(0);
        const rect = range.getBoundingClientRect();

        selectionPopover = createSelectionPopover(rect, selectedText);
        document.body.appendChild(selectionPopover);

        // Remove on click elsewhere
        removePopoverHandler = (evt) => {
          if (!evt.target.closest('#haiven-selection-popover')) {
            if (selectionPopover) {
              selectionPopover.remove();
              selectionPopover = null;
            }
            document.removeEventListener('mousedown', removePopoverHandler);
            removePopoverHandler = null;
          }
        };

        setTimeout(() => {
          document.addEventListener('mousedown', removePopoverHandler);
        }, 100);
      } catch (err) {
        // Selection might be invalid
        console.debug('Haiven: Could not get selection range', err);
      }
    }

    // Listen for text selection with cleanup
    cleanupManager.addListener(document, 'mouseup', handleTextSelection);

    // ========================================
    // END-OF-CONVERSATION SUMMARY (Option D)
    // Prompts user to save key insights when leaving
    // ========================================

    const conversationTracker = {
      pairs: [],
      currentUrl: window.location.href,
      sessionStart: Date.now(),
      hasShownPrompt: false,
      minPairsForPrompt: 2, // At least 2 Q&A pairs before prompting

      // Track a Q&A pair
      addPair(question, answer) {
        // Skip if too short
        if (question.length < 20 || answer.length < 30) return;

        const questionTrimmed = question.substring(0, 500);
        const answerTrimmed = answer.substring(0, 1000);

        // DEDUPLICATION: Check if this exact pair already exists
        const isDuplicate = this.pairs.some(
          (p) => p.question === questionTrimmed && p.answer === answerTrimmed
        );

        if (isDuplicate) {
          return; // Skip duplicate
        }

        this.pairs.push({
          question: questionTrimmed,
          answer: answerTrimmed,
          timestamp: Date.now()
        });

        // Keep only last 10 pairs to avoid memory bloat
        if (this.pairs.length > 10) {
          this.pairs.shift();
        }

        console.log(`🧠 Haiven: Tracked Q&A pair (${this.pairs.length} total)`);
      },

      // Check if we have enough content to prompt
      shouldPrompt() {
        return (
          !this.hasShownPrompt &&
          this.pairs.length >= this.minPairsForPrompt &&
          Date.now() - this.sessionStart > 30000
        ); // At least 30 seconds
      },

      // Extract key insights from conversation
      extractInsights() {
        if (this.pairs.length === 0) return [];

        const insights = [];

        // Look for answers that contain actionable info
        const actionPatterns = [
          /you (should|can|need to|must|could)/i,
          /the (solution|answer|way|method|approach) is/i,
          /try (using|doing|this)/i,
          /here's (how|what|the)/i,
          /important[:\s]/i,
          /key (point|takeaway|insight)/i
        ];

        this.pairs.forEach((pair, _idx) => {
          // Check if answer has useful patterns
          const hasAction = actionPatterns.some((p) => p.test(pair.answer));

          // Check for code blocks
          const hasCode = /```[\s\S]+```/.test(pair.answer);

          // Check for lists (numbered or bulleted)
          const hasList = /^[\s]*[-•*]|\d+\./m.test(pair.answer);

          if (hasAction || hasCode || hasList) {
            insights.push({
              type: hasCode ? 'code' : hasList ? 'list' : 'insight',
              question: pair.question,
              answer: pair.answer,
              preview: this.createPreview(pair.answer, 150)
            });
          }
        });

        // If no specific insights found, use last 2 pairs
        if (insights.length === 0 && this.pairs.length >= 2) {
          const lastPairs = this.pairs.slice(-2);
          lastPairs.forEach((pair) => {
            insights.push({
              type: 'conversation',
              question: pair.question,
              answer: pair.answer,
              preview: this.createPreview(pair.answer, 150)
            });
          });
        }

        return insights.slice(0, 5); // Max 5 insights
      },

      createPreview(text, maxLen) {
        // Remove code blocks for preview
        let preview = text.replace(/```[\s\S]*?```/g, '[code]');
        // Remove markdown formatting
        preview = preview.replace(/[*_`#]/g, '');
        // Truncate
        if (preview.length > maxLen) {
          preview = preview.substring(0, maxLen).trim() + '...';
        }
        return preview;
      },

      reset() {
        this.pairs = [];
        this.currentUrl = window.location.href;
        this.sessionStart = Date.now();
        this.hasShownPrompt = false;
      }
    };

    // Monitor for new messages to track Q&A pairs
    function setupConversationTracking() {
      let lastSeenMessages = 0;

      const checkForNewMessages = () => {
        const userMsgs = document.querySelectorAll(currentPlatform.messageSelector);
        const assistantMsgs = document.querySelectorAll(currentPlatform.assistantSelector);

        // Only process if we have new messages
        if (userMsgs.length + assistantMsgs.length <= lastSeenMessages) return;
        lastSeenMessages = userMsgs.length + assistantMsgs.length;

        // Try to pair last user message with last assistant message
        if (userMsgs.length > 0 && assistantMsgs.length > 0) {
          const lastUser = userMsgs[userMsgs.length - 1];
          const lastAssistant = assistantMsgs[assistantMsgs.length - 1];

          if (lastUser && lastAssistant) {
            const question = lastUser.textContent || lastUser.innerText || '';
            const answer = lastAssistant.textContent || lastAssistant.innerText || '';
            conversationTracker.addPair(question, answer);
          }
        }
      };

      // Check periodically
      const trackingInterval = cleanupManager.addInterval(setInterval(checkForNewMessages, 3000));

      return trackingInterval;
    }

    // Show conversation summary modal with GPT-powered analysis
    async function showConversationSummaryModal() {
      if (conversationTracker.hasShownPrompt) return;
      conversationTracker.hasShownPrompt = true;

      // Check if we have enough conversation content
      if (conversationTracker.pairs.length < 2) {
        console.log('🧠 Haiven: Not enough conversation to analyze');
        return;
      }

      // Create modal overlay with loading state
      const overlay = createElement('div', {
        id: 'haiven-summary-overlay',
        style: {
          position: 'fixed',
          inset: '0',
          background: 'rgba(0, 0, 0, 0.8)',
          backdropFilter: 'blur(8px)',
          display: 'flex',
          alignItems: 'center',
          justifyContent: 'center',
          zIndex: '1000001',
          animation: 'haivenFadeIn 0.3s ease-out'
        }
      });

      // Create modal content
      const modal = createElement('div', {
        style: {
          background: 'linear-gradient(135deg, #1a1a2e 0%, #16213e 100%)',
          borderRadius: '16px',
          padding: '24px',
          maxWidth: '520px',
          width: '90%',
          maxHeight: '80vh',
          overflow: 'hidden',
          display: 'flex',
          flexDirection: 'column',
          boxShadow: '0 20px 60px rgba(0, 0, 0, 0.5), 0 0 0 1px rgba(0, 212, 255, 0.2)',
          animation: 'haivenSlideUp 0.3s ease-out'
        }
      });

      // Loading state
      const loadingContent = createElement(
        'div',
        {
          id: 'haiven-loading-content',
          style: {
            textAlign: 'center',
            padding: '40px 20px'
          }
        },
        [
          createElement(
            'div',
            {
              style: {
                fontSize: '40px',
                marginBottom: '16px',
                animation: 'haivenPulse 1.5s ease-in-out infinite'
              }
            },
            ['🧠']
          ),
          createElement(
            'h3',
            {
              style: {
                margin: '0 0 8px 0',
                fontSize: '18px',
                fontWeight: '600',
                color: '#fff'
              }
            },
            ['Analyzing Conversation...']
          ),
          createElement(
            'p',
            {
              style: {
                margin: '0',
                fontSize: '13px',
                color: 'rgba(255, 255, 255, 0.6)'
              }
            },
            ['Finding the most valuable insights to save']
          )
        ]
      );

      modal.appendChild(loadingContent);
      overlay.appendChild(modal);
      document.body.appendChild(overlay);

      // Add pulse animation if not present
      if (!document.getElementById('haiven-pulse-animation')) {
        const style = createElement('style', { id: 'haiven-pulse-animation' });
        style.textContent = `
          @keyframes haivenPulse { 0%, 100% { transform: scale(1); opacity: 1; } 50% { transform: scale(1.1); opacity: 0.8; } }
        `;
        document.head.appendChild(style);
      }

      // Build conversation text for analysis (reserved for GPT analysis)
      const _conversationText = conversationTracker.pairs
        .map((p) => `User: ${p.question}\n\nAssistant: ${p.answer}`)
        .join('\n\n---\n\n');

      // Call GPT-powered analysis via background script
      let suggestions = [];
      let analysisResult = null;

      try {
        const response = await new Promise((resolve, reject) => {
          browserAPI.runtime.sendMessage(
            {
              type: 'ANALYZE_CONVERSATION',
              data: {
                messages: conversationTracker.pairs
                  .map((p) => [
                    { role: 'user', content: p.question },
                    { role: 'assistant', content: p.answer }
                  ])
                  .flat(),
                platform: currentPlatform.name,
                url: window.location.href
              }
            },
            (response) => {
              if (response?.success) {
                resolve(response.data);
              } else {
                reject(new Error(response?.error || 'Analysis failed'));
              }
            }
          );
        });

        analysisResult = response;
        suggestions = response.suggestions || [];
      } catch (error) {
        console.error('🧠 Haiven: GPT analysis failed, using fallback:', error);
        // Fallback to basic pattern extraction
        const fallbackInsights = conversationTracker.extractInsights();
        suggestions = fallbackInsights.map((insight) => ({
          summary: insight.preview,
          category: insight.type,
          why_save: 'Extracted from conversation',
          confidence: 0.7,
          isDuplicate: false,
          fullContent: `Q: ${insight.question}\n\nA: ${insight.answer}`
        }));
      }

      // Remove loading, show results
      loadingContent.remove();

      if (suggestions.length === 0) {
        // No valuable insights found
        const emptyContent = createElement(
          'div',
          {
            style: { textAlign: 'center', padding: '30px 20px' }
          },
          [
            createElement('div', { style: { fontSize: '40px', marginBottom: '12px' } }, ['🤷']),
            createElement(
              'h3',
              {
                style: { margin: '0 0 8px 0', fontSize: '18px', color: '#fff' }
              },
              ['Nothing Worth Saving']
            ),
            createElement(
              'p',
              {
                style: { margin: '0 0 20px 0', fontSize: '13px', color: 'rgba(255, 255, 255, 0.6)' }
              },
              ["This conversation didn't contain decisions, code, or actionable insights."]
            ),
            createElement(
              'button',
              {
                style: {
                  padding: '10px 24px',
                  borderRadius: '8px',
                  border: 'none',
                  background: 'rgba(255, 255, 255, 0.1)',
                  color: '#fff',
                  fontSize: '14px',
                  cursor: 'pointer'
                },
                onClick: () => overlay.remove()
              },
              ['Got it']
            )
          ]
        );
        modal.appendChild(emptyContent);
        return;
      }

      // Category icons
      const categoryIcons = {
        decision: '🎯',
        code: '💻',
        advice: '💡',
        todo: '✅',
        fact: '📚',
        insight: '🔮',
        preference: '⭐',
        list: '📋',
        conversation: '💬'
      };

      // Header
      const header = createElement(
        'div',
        {
          style: {
            display: 'flex',
            alignItems: 'center',
            justifyContent: 'space-between',
            marginBottom: '16px',
            paddingBottom: '16px',
            borderBottom: '1px solid rgba(255, 255, 255, 0.1)'
          }
        },
        [
          createElement(
            'div',
            {
              style: { display: 'flex', alignItems: 'center', gap: '12px' }
            },
            [
              createElement('span', { style: { fontSize: '28px' } }, ['🧠']),
              createElement('div', {}, [
                createElement(
                  'h3',
                  {
                    style: { margin: '0', fontSize: '18px', fontWeight: '600', color: '#fff' }
                  },
                  ['Smart Save']
                ),
                createElement(
                  'p',
                  {
                    style: {
                      margin: '4px 0 0 0',
                      fontSize: '13px',
                      color: 'rgba(255, 255, 255, 0.6)'
                    }
                  },
                  [
                    analysisResult?.topic ? `${analysisResult.topic} • ` : '',
                    `${suggestions.length} insight${suggestions.length !== 1 ? 's' : ''} found`
                  ]
                )
              ])
            ]
          ),
          createElement(
            'button',
            {
              style: {
                background: 'none',
                border: 'none',
                color: 'rgba(255, 255, 255, 0.5)',
                fontSize: '24px',
                cursor: 'pointer',
                padding: '4px',
                lineHeight: '1'
              },
              onClick: () => overlay.remove()
            },
            ['×']
          )
        ]
      );

      // Insights list
      const insightsList = createElement('div', {
        style: {
          flex: '1',
          overflowY: 'auto',
          marginBottom: '16px',
          maxHeight: '350px'
        }
      });

      // Track selected insights (exclude duplicates by default)
      const selectedInsights = new Set();
      suggestions.forEach((s, i) => {
        if (!s.isDuplicate) selectedInsights.add(i);
      });

      // Note: saveBtn is accessed via getElementById in updateSaveButton for forward reference

      suggestions.forEach((suggestion, idx) => {
        const icon = categoryIcons[suggestion.category] || '💡';
        const isDupe = suggestion.isDuplicate;

        const insightEl = createElement(
          'label',
          {
            style: {
              display: 'flex',
              alignItems: 'flex-start',
              gap: '12px',
              padding: '12px',
              marginBottom: '8px',
              background: isDupe ? 'rgba(255, 200, 0, 0.05)' : 'rgba(255, 255, 255, 0.05)',
              borderRadius: '8px',
              cursor: 'pointer',
              transition: 'background 0.2s',
              border: isDupe
                ? '1px solid rgba(255, 200, 0, 0.3)'
                : '1px solid rgba(255, 255, 255, 0.1)',
              opacity: isDupe ? '0.7' : '1'
            }
          },
          [
            createElement('input', {
              type: 'checkbox',
              checked: selectedInsights.has(idx) ? 'checked' : null,
              style: {
                width: '18px',
                height: '18px',
                marginTop: '2px',
                accentColor: '#00d4ff'
              },
              onChange: (e) => {
                if (e.target.checked) {
                  selectedInsights.add(idx);
                } else {
                  selectedInsights.delete(idx);
                }
                updateSaveButton();
              }
            }),
            createElement('div', { style: { flex: '1', minWidth: '0' } }, [
              createElement(
                'div',
                {
                  style: {
                    display: 'flex',
                    alignItems: 'center',
                    gap: '6px',
                    marginBottom: '4px',
                    flexWrap: 'wrap'
                  }
                },
                [
                  createElement('span', { style: { fontSize: '14px' } }, [icon]),
                  createElement(
                    'span',
                    {
                      style: {
                        fontSize: '11px',
                        color: '#00d4ff',
                        textTransform: 'uppercase',
                        fontWeight: '600'
                      }
                    },
                    [suggestion.category]
                  ),
                  createElement(
                    'span',
                    {
                      style: {
                        fontSize: '11px',
                        color: 'rgba(255, 255, 255, 0.4)',
                        marginLeft: '4px'
                      }
                    },
                    [`${Math.round((suggestion.confidence || 0.7) * 100)}%`]
                  ),
                  ...(isDupe
                    ? [
                        createElement(
                          'span',
                          {
                            style: {
                              fontSize: '10px',
                              background: 'rgba(255, 200, 0, 0.2)',
                              color: '#ffcc00',
                              padding: '2px 6px',
                              borderRadius: '4px',
                              marginLeft: '4px'
                            }
                          },
                          ['Already saved']
                        )
                      ]
                    : [])
                ]
              ),
              createElement(
                'p',
                {
                  style: {
                    margin: '0',
                    fontSize: '13px',
                    color: 'rgba(255, 255, 255, 0.8)',
                    lineHeight: '1.4',
                    wordBreak: 'break-word'
                  }
                },
                [suggestion.summary]
              ),
              ...(suggestion.why_save
                ? [
                    createElement(
                      'p',
                      {
                        style: {
                          margin: '6px 0 0 0',
                          fontSize: '11px',
                          color: 'rgba(255, 255, 255, 0.5)',
                          fontStyle: 'italic'
                        }
                      },
                      [`💬 ${suggestion.why_save}`]
                    )
                  ]
                : [])
            ])
          ]
        );

        insightsList.appendChild(insightEl);
      });

      // Footer
      const footer = createElement('div', {
        style: {
          display: 'flex',
          gap: '12px',
          justifyContent: 'flex-end',
          paddingTop: '16px',
          borderTop: '1px solid rgba(255, 255, 255, 0.1)'
        }
      });

      const skipBtn = createElement(
        'button',
        {
          style: {
            padding: '10px 20px',
            borderRadius: '8px',
            border: '1px solid rgba(255, 255, 255, 0.2)',
            background: 'transparent',
            color: 'rgba(255, 255, 255, 0.7)',
            fontSize: '14px',
            fontWeight: '500',
            cursor: 'pointer'
          },
          onClick: () => overlay.remove()
        },
        ['Skip']
      );

      const saveBtn = createElement(
        'button',
        {
          id: 'haiven-save-insights-btn',
          style: {
            padding: '10px 24px',
            borderRadius: '8px',
            border: 'none',
            background:
              selectedInsights.size > 0
                ? 'linear-gradient(135deg, #00d4ff, #0099ff)'
                : 'rgba(255, 255, 255, 0.1)',
            color: '#fff',
            fontSize: '14px',
            fontWeight: '600',
            cursor: 'pointer',
            display: 'flex',
            alignItems: 'center',
            gap: '8px',
            opacity: selectedInsights.size > 0 ? '1' : '0.5'
          },
          onClick: async () => {
            if (selectedInsights.size === 0) return;

            const btn = document.getElementById('haiven-save-insights-btn');
            if (btn) {
              btn.disabled = true;
              btn.textContent = 'Saving...';
            }

            const toSave = suggestions
              .filter((_, i) => selectedInsights.has(i))
              .map((s) => ({
                content: s.fullContent || s.summary,
                category: s.category,
                confidence: s.confidence,
                why_save: s.why_save,
                tags: [s.category]
              }));

            try {
              const response = await new Promise((resolve) => {
                browserAPI.runtime.sendMessage(
                  {
                    type: 'BATCH_SAVE_MEMORIES',
                    data: {
                      memories: toSave,
                      platform: currentPlatform.name,
                      url: window.location.href
                    }
                  },
                  resolve
                );
              });

              overlay.remove();

              if (response?.success) {
                showNotification(
                  `✅ Saved ${response.data?.savedCount || toSave.length} memories to Haiven`,
                  'success'
                );
              } else {
                showNotification('⚠️ Some memories may not have saved', 'warning');
              }
            } catch (err) {
              console.error('🧠 Haiven: Batch save failed:', err);
              overlay.remove();
              showNotification('❌ Failed to save memories', 'error');
            }
          }
        },
        [`Save ${selectedInsights.size} Memory${selectedInsights.size !== 1 ? 'ies' : 'y'}`]
      );

      const updateSaveButton = () => {
        saveBtn.textContent = `Save ${selectedInsights.size} Memory${selectedInsights.size !== 1 ? 'ies' : 'y'}`;
        saveBtn.disabled = selectedInsights.size === 0;
        saveBtn.style.opacity = selectedInsights.size === 0 ? '0.5' : '1';
        saveBtn.style.background =
          selectedInsights.size > 0
            ? 'linear-gradient(135deg, #00d4ff, #0099ff)'
            : 'rgba(255, 255, 255, 0.1)';
      };

      footer.appendChild(skipBtn);
      footer.appendChild(saveBtn);

      modal.appendChild(header);
      modal.appendChild(insightsList);
      modal.appendChild(footer);

      // Close on overlay click
      overlay.addEventListener('click', (e) => {
        if (e.target === overlay) overlay.remove();
      });
    }

    // Detect when user is leaving the conversation
    function setupConversationEndDetection() {
      // Track URL changes (user navigates to different chat)
      let lastUrl = window.location.href;

      const checkUrlChange = () => {
        const currentUrl = window.location.href;
        if (currentUrl !== lastUrl) {
          // URL changed - potentially new conversation
          if (conversationTracker.shouldPrompt()) {
            showConversationSummaryModal();
          }
          // Reset tracker for new conversation
          conversationTracker.reset();
          lastUrl = currentUrl;
        }
      };

      // Check URL periodically (SPA navigation)
      cleanupManager.addInterval(setInterval(checkUrlChange, 1000));

      // Handle visibility change (user switching tabs/windows)
      cleanupManager.addListener(document, 'visibilitychange', () => {
        if (document.visibilityState === 'hidden') {
          // User is leaving - show prompt if appropriate
          if (conversationTracker.shouldPrompt()) {
            // Delay slightly to avoid interrupting quick tab switches
            setTimeout(() => {
              if (document.visibilityState === 'hidden') {
                // Store flag to show on return
                conversationTracker.pendingPrompt = true;
              }
            }, 2000);
          }
        } else if (document.visibilityState === 'visible' && conversationTracker.pendingPrompt) {
          // User returned - show prompt now
          conversationTracker.pendingPrompt = false;
          showConversationSummaryModal();
        }
      });

      // Handle page unload (closing tab)
      cleanupManager.addListener(window, 'beforeunload', (_e) => {
        if (conversationTracker.shouldPrompt()) {
          // Try to show prompt (may not work in all browsers)
          showConversationSummaryModal();
        }
      });
    }

    // Initialize conversation tracking
    setupConversationTracking();
    setupConversationEndDetection();

    console.log('🧠 Haiven: Conversation summary tracking enabled');

    // ========================================
    // INITIALIZE
    // ========================================

    console.log('🧠 Haiven: Extension loaded for', currentPlatform.name);

    // Initialize extension - load settings first to prevent race conditions
    (async function initExtension() {
      // CRITICAL: Load settings BEFORE everything else
      await loadSettings();

      // Show morning briefing on page load (after 2 seconds)
      setTimeout(() => {
        showMorningBriefing();
      }, 2000);

      // Start auto-capture
      await startAutoCapture();

      // Start proactive intelligence monitoring
      await startProactiveMonitoring();

      // Initialize Waste Detector (shows users their invisible waste)
      if (window.HaivenWasteDetector) {
        setTimeout(() => {
          window.HaivenWasteDetector.init(currentPlatform);
        }, 1000);
      }
    })();

    // Inject Haiven button (inline + floating backup)
    // Initial injection after page loads
    setTimeout(() => {
      injectHaivenButton();
    }, 2000);

    // Re-inject on DOM changes (for SPAs like ChatGPT that dynamically render)
    const buttonObserver = new MutationObserver(() => {
      // Only re-inject if button doesn't exist
      if (
        !document.getElementById('haiven-inline-btn') &&
        !document.getElementById('haiven-floating-btn')
      ) {
        injectHaivenButton();
      }
    });

    // Start observing after initial load
    setTimeout(() => {
      buttonObserver.observe(document.body, {
        childList: true,
        subtree: true
      });
    }, 3000);

    // Listen for storage changes (if user toggles auto-capture)
    browserAPI.storage.onChanged.addListener((changes, area) => {
      if (area === 'local' && changes.autoCapture) {
        console.log('🧠 Haiven: Auto-capture toggled:', changes.autoCapture.newValue);
        if (changes.autoCapture.newValue && !observerActive) {
          startAutoCapture();
        }
      }
    });

    // ========================================
    // 🆕 FEATURE 1: CONTEXT CAPTURE
    // Capture page context (URL, title, selected text)
    // ========================================

    browserAPI.runtime.onMessage.addListener((request, sender, sendResponse) => {
      if (request.action === 'getPageContext') {
        const context = {
          url: window.location.href,
          title: document.title,
          selectedText: window.getSelection().toString().trim(),
          domain: window.location.hostname,
          timestamp: new Date().toISOString()
        };

        console.log('📄 Context captured:', context);
        sendResponse(context);
        return true; // Required for async response
      }
    });

    // ========================================
    // 🆕 FEATURE 2: SAVE GENERATED CONTENT
    // Add "Save to Haiven" buttons on generated images, code artifacts
    // ========================================

    // Platform-specific selectors for generated content
    const GENERATED_CONTENT_SELECTORS = {
      ChatGPT: {
        images:
          '[data-testid="image-container"] img, .dall-e-image img, img[alt*="Generated"], [data-testid="generated-image"]',
        code: 'pre code, .code-block, [data-testid="code-block"]',
        artifacts: '.artifact-container, [data-testid="artifact"]'
      },
      Claude: {
        images: '.artifact-image img, [data-testid="artifact-image"], .generated-image',
        code: '.artifact-code, [data-testid="code-artifact"], pre code',
        artifacts: '[data-testid="artifact-container"], .artifact-panel'
      },
      Gemini: {
        images: '.generated-image img, [data-testid="imagen-output"], .model-response img',
        code: 'pre code, .code-block',
        artifacts: '.artifact'
      }
    };

    // Create "Save to Haiven" button element
    function createSaveToHaivenButton(type, targetElement) {
      const btn = createElement(
        'button',
        {
          className: 'haiven-save-content-btn',
          dataset: { contentType: type },
          style: {
            position: 'absolute',
            top: '8px',
            right: '8px',
            padding: '6px 12px',
            fontSize: '12px',
            fontWeight: '600',
            color: '#fff',
            background: 'linear-gradient(135deg, #f59e0b, #0ea5e9)',
            border: 'none',
            borderRadius: '6px',
            cursor: 'pointer',
            zIndex: '10000',
            display: 'flex',
            alignItems: 'center',
            gap: '4px',
            opacity: '0',
            transition: 'opacity 0.2s ease',
            boxShadow: '0 2px 8px rgba(245, 158, 11, 0.3)'
          },
          onClick: async (e) => {
            e.preventDefault();
            e.stopPropagation();
            await saveContentToHaiven(type, targetElement, btn);
          }
        },
        [
          createElement('span', { textContent: '📎' }),
          createElement('span', { textContent: 'Save to Haiven' })
        ]
      );

      return btn;
    }

    // Save content to Haiven
    async function saveContentToHaiven(type, element, button) {
      if (!USER_ID) {
        showModal({
          title: 'Not Logged In',
          message: 'Please configure your Haiven user ID in the extension settings first.',
          type: 'warning'
        });
        return;
      }

      const originalText = button.querySelector('span:last-child').textContent;
      button.querySelector('span:last-child').textContent = 'Saving...';
      button.style.opacity = '0.7';
      button.disabled = true;

      try {
        let content = '';
        const contentType = type;
        let imageData = null;

        if (type === 'image') {
          // Get the image source
          const img = element.tagName === 'IMG' ? element : element.querySelector('img');
          if (img && img.src) {
            // Try to convert image to base64
            try {
              const response = await fetch(img.src);
              const blob = await response.blob();
              imageData = await blobToBase64(blob);
              content = `Generated image from ${currentPlatform.name}`;
            } catch (imgError) {
              // Fallback to just URL
              content = `Generated image: ${img.src}`;
              console.warn('Could not fetch image data:', imgError);
            }
          }
        } else if (type === 'code') {
          // Get code content
          const codeEl = element.tagName === 'CODE' ? element : element.querySelector('code');
          content = codeEl ? codeEl.textContent : element.textContent;
          content = `Code snippet from ${currentPlatform.name}:\n\n\`\`\`\n${content}\n\`\`\``;
        } else if (type === 'artifact') {
          // Get artifact content
          content = element.textContent || element.innerText;
          content = `Artifact from ${currentPlatform.name}:\n\n${content}`;
        }

        // If we have image data, use the upload endpoint
        if (imageData) {
          const headers = await getAuthHeaders();
          const uploadResponse = await safeFetch(`${API_BASE}/api/upload/image`, {
            method: 'POST',
            headers,
            body: JSON.stringify({
              userId: USER_ID,
              image: imageData,
              source: currentPlatform.name,
              metadata: {
                type: 'generated_image',
                platform: currentPlatform.name,
                url: window.location.href,
                timestamp: new Date().toISOString()
              }
            })
          });

          if (!uploadResponse.ok) {
            throw new Error('Failed to upload image');
          }

          button.querySelector('span:first-child').textContent = '✅';
          button.querySelector('span:last-child').textContent = 'Saved!';
          console.log('📸 Image saved to Haiven');
        } else if (content) {
          // Save as text memory
          const headers = await getAuthHeaders();
          const storeResponse = await safeFetch(`${API_BASE}/api/memory/store/fast`, {
            method: 'POST',
            headers,
            body: JSON.stringify({
              userId: USER_ID,
              content: content.substring(0, 5000), // Limit content length
              source: currentPlatform.name,
              metadata: {
                type: contentType,
                platform: currentPlatform.name,
                url: window.location.href
              }
            })
          });

          if (!storeResponse.ok) {
            throw new Error('Failed to save content');
          }

          button.querySelector('span:first-child').textContent = '✅';
          button.querySelector('span:last-child').textContent = 'Saved!';
          console.log(`💾 ${type} saved to Haiven`);
        }

        // Reset button after delay
        setTimeout(() => {
          button.querySelector('span:first-child').textContent = '📎';
          button.querySelector('span:last-child').textContent = originalText;
          button.style.opacity = '0';
          button.disabled = false;
        }, 2000);
      } catch (error) {
        console.error('Failed to save to Haiven:', error);
        button.querySelector('span:first-child').textContent = '❌';
        button.querySelector('span:last-child').textContent = 'Failed';
        button.style.opacity = '1';

        setTimeout(() => {
          button.querySelector('span:first-child').textContent = '📎';
          button.querySelector('span:last-child').textContent = originalText;
          button.style.opacity = '0';
          button.disabled = false;
        }, 2000);
      }
    }

    // Convert blob to base64
    function blobToBase64(blob) {
      return new Promise((resolve, reject) => {
        const reader = new FileReader();
        reader.onloadend = () => resolve(reader.result);
        reader.onerror = reject;
        reader.readAsDataURL(blob);
      });
    }

    // Inject save buttons on generated content
    function injectSaveButtons() {
      const platformSelectors = GENERATED_CONTENT_SELECTORS[currentPlatform.name];
      if (!platformSelectors) return;

      // Process images
      if (platformSelectors.images) {
        document.querySelectorAll(platformSelectors.images).forEach((img) => {
          if (img.closest('.haiven-save-wrapper')) return; // Already wrapped
          wrapWithSaveButton(img, 'image');
        });
      }

      // Process code blocks
      if (platformSelectors.code) {
        document.querySelectorAll(platformSelectors.code).forEach((code) => {
          if (code.closest('.haiven-save-wrapper')) return;
          // Only wrap if code has substantial content
          if (code.textContent && code.textContent.length > 20) {
            wrapWithSaveButton(code, 'code');
          }
        });
      }

      // Process artifacts
      if (platformSelectors.artifacts) {
        document.querySelectorAll(platformSelectors.artifacts).forEach((artifact) => {
          if (artifact.closest('.haiven-save-wrapper')) return;
          wrapWithSaveButton(artifact, 'artifact');
        });
      }
    }

    // Wrap element with save button container
    function wrapWithSaveButton(element, type) {
      // Check if element or parent is already wrapped
      if (element.closest('.haiven-save-wrapper')) return;

      const wrapper = createElement('div', {
        className: 'haiven-save-wrapper',
        style: {
          position: 'relative',
          display: 'inline-block'
        }
      });

      // Clone to avoid breaking the original element
      const parent = element.parentNode;
      if (!parent) return;

      // Wrap the element
      parent.insertBefore(wrapper, element);
      wrapper.appendChild(element);

      // Add save button
      const saveBtn = createSaveToHaivenButton(type, element);
      wrapper.appendChild(saveBtn);

      // Show button on hover
      wrapper.addEventListener('mouseenter', () => {
        saveBtn.style.opacity = '1';
      });
      wrapper.addEventListener('mouseleave', () => {
        if (!saveBtn.disabled) {
          saveBtn.style.opacity = '0';
        }
      });
    }

    // Check settings before initializing save buttons
    async function initSaveButtons() {
      try {
        const settings = await browserAPI.storage.local.get([
          'showSaveOnImages',
          'showSaveOnCode',
          'showSaveOnArtifacts'
        ]);

        // Default to true if not set
        const showOnImages = settings.showSaveOnImages !== false;
        const showOnCode = settings.showSaveOnCode !== false;
        const showOnArtifacts = settings.showSaveOnArtifacts !== false;

        if (showOnImages || showOnCode || showOnArtifacts) {
          // Initial injection
          setTimeout(injectSaveButtons, 2000);

          // Watch for new content
          const saveButtonObserver = new MutationObserver((mutations) => {
            // Debounce
            clearTimeout(saveButtonObserver._timeout);
            saveButtonObserver._timeout = setTimeout(injectSaveButtons, 500);
          });

          saveButtonObserver.observe(document.body, {
            childList: true,
            subtree: true
          });

          cleanupManager.addObserver(saveButtonObserver);
          console.log('📎 Haiven: Save buttons enabled for generated content');
        }
      } catch (error) {
        console.error('Failed to initialize save buttons:', error);
      }
    }

    // Initialize save buttons feature
    initSaveButtons();
  } catch (error) {
    console.error('❌ Haiven Extension Error:', error);
    console.error('Stack:', error.stack);
  }
})();
