/* global browser, chrome */
// Browser compatibility layer
const browserAPI = typeof browser !== 'undefined' ? browser : chrome;
// Haiven Memory OS - Background Service Worker
// Handles all API communication and memory capture

const HAIVEN_API = 'https://api.safehaiven.com';
let captureCount = 0;

// Auth helper - Get headers with API key or userId as fallback
async function getAuthHeaders() {
  const { apiKey, userId } = await browserAPI.storage.local.get(['apiKey', 'userId']);
  const headers = {
    'Content-Type': 'application/json'
  };
  // Use API key if available, otherwise use userId as Bearer token
  if (apiKey) {
    headers['Authorization'] = `Bearer ${apiKey}`;
  } else if (userId) {
    headers['Authorization'] = `Bearer ${userId}`;
  }
  return headers;
}

// Listen for messages from content scripts
browserAPI.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'CAPTURE_MEMORY') {
    captureMemory(message.data)
      .then((response) => sendResponse({ success: true, data: response }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true; // Keep channel open for async response
  }

  if (message.type === 'GET_USER_ID') {
    getUserId()
      .then((userId) => sendResponse({ success: true, userId }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }

  if (message.action === 'captureMemory') {
    // Legacy support
    captureMemory(message.data)
      .then(() => sendResponse({ success: true }))
      .catch(() => sendResponse({ success: false }));
    return true;
  }

  if (message.action === 'openTab') {
    browserAPI.tabs.create({ url: message.url });
    sendResponse({ success: true });
    return false;
  }

  // NEW: Handle conversation decomposition
  if (message.action === 'decompose-conversation') {
    decomposeConversation(message)
      .then((response) => sendResponse({ success: true, data: response }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }

  // SMART END-OF-CHAT: Analyze conversation for valuable memories
  if (message.type === 'ANALYZE_CONVERSATION') {
    analyzeConversationForMemories(message.data)
      .then((response) => sendResponse({ success: true, data: response }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }

  // CHECK DUPLICATES: Check if memory already exists
  if (message.type === 'CHECK_DUPLICATE') {
    checkForDuplicateMemory(message.data)
      .then((response) => sendResponse({ success: true, data: response }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }

  // BATCH SAVE: Save multiple memories at once
  if (message.type === 'BATCH_SAVE_MEMORIES') {
    batchSaveMemories(message.data)
      .then((response) => sendResponse({ success: true, data: response }))
      .catch((error) => sendResponse({ success: false, error: error.message }));
    return true;
  }
});

// Capture memory via Haiven API
async function captureMemory(memoryData) {
  const userId = await getUserId();

  if (!userId) {
    throw new Error('User not set up. Please configure extension settings.');
  }

  // Check if auto-capture is enabled
  const { autoCapture } = await browserAPI.storage.local.get(['autoCapture']);
  if (autoCapture === false && !memoryData.explicit) {
    console.log('Auto-capture disabled, skipping');
    return { skipped: true, reason: 'Auto-capture disabled' };
  }

  // P1.5: Auto-generate tags if not provided
  const autoTags = autoGenerateTags(memoryData.content);
  const allTags = [...new Set([...(memoryData.tags || []), ...autoTags])];

  // P0.1: Detect content type
  const contentType = detectContentType(memoryData.content);

  let response;
  try {
    response = await fetch(`${HAIVEN_API}/api/memory/store/fast`, {
      method: 'POST',
      headers: await getAuthHeaders(),
      body: JSON.stringify({
        userId,
        content: memoryData.content,
        source: memoryData.source || 'browser-extension',
        tags: allTags,
        world_name: memoryData.world_name || null,
        // IMPORTANT: Bypass filter for explicit user saves (right-click, keyboard shortcut, etc.)
        bypassFilter: memoryData.explicit || false,
        trigger: memoryData.explicit ? 'user-explicit-save' : null,
        metadata: {
          url: memoryData.url || '',
          timestamp: new Date().toISOString(),
          platform: memoryData.platform || 'unknown',
          explicit: memoryData.explicit || false,
          contentType: contentType,
          wordCount: memoryData.content.split(/\s+/).length
        }
      })
    });

    if (!response.ok) {
      const errorBody = await response.text().catch(() => 'Unable to parse error');
      let errorMessage = `API error: ${response.status}`;

      try {
        const errorJson = JSON.parse(errorBody);
        errorMessage = errorJson.error || errorJson.message || errorMessage;
      } catch {
        // Error body is not JSON
        if (errorBody) {
          errorMessage = `${errorMessage} - ${errorBody.substring(0, 100)}`;
        }
      }

      // Handle specific error codes
      if (response.status === 429) {
        throw new Error('Rate limit exceeded. Please try again later.');
      } else if (response.status === 401 || response.status === 403) {
        throw new Error('Authentication failed. Please check your User ID in settings.');
      } else if (response.status >= 500) {
        throw new Error('Server error. Your memory will be retried automatically.');
      }

      throw new Error(errorMessage);
    }
  } catch (error) {
    // Network errors
    if (error.name === 'TypeError' && error.message.includes('fetch')) {
      throw new Error('Network error. Please check your internet connection.');
    }
    throw error;
  }

  const result = await response.json();

  // Check if server actually saved the memory (it can reject low-quality content)
  if (result.success === false || result.rejected) {
    console.log('🧠 Haiven: Memory rejected by server:', result.reason || 'Unknown reason');

    // Show rejection notification for explicit saves
    if (memoryData.explicit) {
      browserAPI.notifications.create({
        type: 'basic',
        iconUrl: 'icon.png',
        title: '⚠️ Not Saved',
        message: result.reason || 'Content filtered as not worth saving',
        priority: 1
      });
    }

    return result;
  }

  // Increment capture count and update badge ONLY on successful save
  captureCount++;
  updateBadge();

  // P0.1: Enhanced capture confirmation with preview
  const wordCount = memoryData.content.split(/\s+/).length;
  const preview = memoryData.content.substring(0, 100);

  browserAPI.notifications.create({
    type: 'basic',
    iconUrl: 'icon.png',
    title: memoryData.explicit ? `✅ Saved: ${contentType}` : `🧠 Auto-Captured: ${contentType}`,
    message: `"${preview}${preview.length < memoryData.content.length ? '...' : ''}"\n\n📊 ${wordCount} words • ${captureCount} captures today`,
    priority: memoryData.explicit ? 2 : 0,
    silent: !memoryData.explicit // Only sound for explicit saves
  });

  return result;
}

// Get user ID from storage
async function getUserId() {
  const result = await browserAPI.storage.local.get(['userId']);
  return result.userId || '';
}

// P1.5: Auto-detect content type for better organization
function detectContentType(content) {
  const lower = content.toLowerCase();
  const wordCount = content.split(/\s+/).length;

  // Structured learning content (roadmaps, tutorials, guides)
  if (
    /(phase|step|stage|part)\s+\d+/i.test(content) ||
    /\d+\.\s+.+[\s\S]*?\d+\./m.test(content) ||
    lower.includes('roadmap') ||
    lower.includes('tutorial')
  ) {
    return '🗺️ Roadmap';
  }

  // Code snippets
  if (
    content.includes('```') ||
    content.includes('function') ||
    content.includes('const ') ||
    content.includes('import ')
  ) {
    return '💻 Code';
  }

  // Action items / todos
  if (/^[-•*]\s+/m.test(content) || lower.includes('action item') || lower.includes('todo')) {
    return '✅ Action Items';
  }

  // Advice / recommendations
  if (lower.includes('i recommend') || lower.includes('you should') || lower.includes('consider')) {
    return '💡 Advice';
  }

  // Long explanations (>500 words)
  if (wordCount > 500) {
    return '📚 Long-form';
  }

  // Research / facts
  if (lower.includes('according to') || lower.includes('research shows')) {
    return '🔬 Research';
  }

  // Default
  if (wordCount > 100) {
    return '📝 Note';
  }

  return '💬 Quick Note';
}

// P1.5: Auto-generate tags from content
function autoGenerateTags(content) {
  const tags = [];
  const lower = content.toLowerCase();

  // Named entities (tools, frameworks, people)
  const entities = {
    figma: 'design',
    react: 'code',
    python: 'code',
    javascript: 'code',
    chatgpt: 'ai',
    claude: 'ai',
    openai: 'ai',
    'ui/ux': 'design',
    ux: 'design',
    ui: 'design'
  };

  for (const [entity, tag] of Object.entries(entities)) {
    if (lower.includes(entity)) {
      tags.push(tag);
    }
  }

  // Content-based tags
  if (/(learn|study|course|tutorial)/i.test(content)) tags.push('learning');
  if (/(work|project|task|deadline)/i.test(content)) tags.push('work');
  if (/(bug|fix|error|issue)/i.test(content)) tags.push('troubleshooting');
  if (/(idea|brainstorm|concept)/i.test(content)) tags.push('ideas');

  return [...new Set(tags)]; // Remove duplicates
}

// Update badge with capture count
function updateBadge() {
  const text = captureCount > 99 ? '99+' : String(captureCount);
  browserAPI.action.setBadgeText({ text });

  // Gradient-like color based on count
  let color = '#f59e0b'; // Default purple
  if (captureCount > 50) {
    color = '#10B981'; // Green for power users
  } else if (captureCount > 20) {
    color = '#1FB6FF'; // Blue for regular users
  }

  browserAPI.action.setBadgeBackgroundColor({ color });
}

// Decompose conversation into multiple atomic memories
async function decomposeConversation(message) {
  const userId = await getUserId();

  if (!userId) {
    throw new Error('User not set up. Please configure extension settings.');
  }

  // P1.5: Auto-generate tags
  const autoTags = autoGenerateTags(message.content);
  const allTags = [...new Set([...(message.tags || []), ...autoTags])];

  // P0.1: Detect content type
  const contentType = detectContentType(message.content);

  const response = await fetch(`${HAIVEN_API}/api/memory/store/fast`, {
    method: 'POST',
    headers: await getAuthHeaders(),
    body: JSON.stringify({
      userId,
      content: message.content,
      source: message.source || 'browser-extension',
      tags: allTags,
      world_name: message.world_name || null,
      metadata: {
        url: message.url || '',
        timestamp: new Date().toISOString(),
        platform: message.platform || 'unknown',
        explicit: message.explicit || false,
        bypassFilter: message.bypassFilter || false,
        trigger: message.trigger || null,
        contextQuestion: message.contextQuestion || null, // P0.3: Conversation context
        contentType: contentType, // P0.1: Content type
        wordCount: message.content.split(/\s+/).length
      }
    })
  });

  if (!response.ok) {
    throw new Error(`API error: ${response.status}`);
  }

  const result = await response.json();

  if (result.success) {
    console.log(`🧠 Haiven: Memory saved successfully`);

    // Update badge
    captureCount++;
    updateBadge();

    // P1.6: Handle duplicate detection
    if (result.action === 'deduplicated') {
      // Show special notification for duplicates
      browserAPI.notifications.create({
        type: 'basic',
        iconUrl: 'icon.png',
        title: '♻️ Already Saved',
        message: `You've mentioned this before! We've noted your ${result.mentionCount || 2}${result.mentionCount === 2 ? 'nd' : result.mentionCount === 3 ? 'rd' : 'th'} mention.\n\n"${message.content.substring(0, 60)}..."`,
        priority: 1
      });
    } else {
      // Show notification if explicit trigger
      if (message.explicit || message.trigger) {
        const contentType = detectContentType(message.content);
        const wordCount = message.content.split(/\s+/).length;

        browserAPI.notifications.create({
          type: 'basic',
          iconUrl: 'icon.png',
          title: `✅ Saved: ${contentType}`,
          message: `"${message.content.substring(0, 70)}..."\n\n📊 ${wordCount} words`,
          priority: 2
        });
      }
    }
  }

  return result;
}

// Reset badge count daily
function resetBadgeDaily() {
  const now = new Date();
  const tomorrow = new Date(now);
  tomorrow.setDate(tomorrow.getDate() + 1);
  tomorrow.setHours(0, 0, 0, 0);

  const timeUntilMidnight = tomorrow - now;

  setTimeout(() => {
    captureCount = 0;
    updateBadge();
    resetBadgeDaily();
  }, timeUntilMidnight);
}

// Initialize on install
browserAPI.runtime.onInstalled.addListener(async () => {
  const { userId } = await browserAPI.storage.local.get(['userId']);

  if (!userId) {
    browserAPI.notifications.create({
      type: 'basic',
      iconUrl: 'icon.png',
      title: 'Haiven Setup Required',
      message: 'Click the extension icon to set up your account',
      priority: 2
    });

    // Set defaults
    browserAPI.storage.local.set({
      autoCapture: true,
      autoSaveEnabled: true,
      autoSaveThreshold: 0.7, // 70% = catches "haiven remind me" patterns
      apiEndpoint: HAIVEN_API
    });
  }

  // Create context menu (right-click) for saving memories
  browserAPI.contextMenus.create({
    id: 'save-to-haiven',
    title: 'Save to Haiven Memory',
    contexts: ['selection']
  });

  updateBadge();
  resetBadgeDaily();
});

// Handle context menu clicks
browserAPI.contextMenus.onClicked.addListener((info, tab) => {
  if (info.menuItemId === 'save-to-haiven' && info.selectionText) {
    captureMemory({
      content: info.selectionText,
      source: 'context-menu',
      url: tab.url,
      explicit: true
    })
      .then(() => {
        browserAPI.notifications.create({
          type: 'basic',
          iconUrl: 'icon.png',
          title: '✅ Saved to Haiven!',
          message: `${info.selectionText.substring(0, 70)}...`,
          priority: 2
        });
      })
      .catch((error) => {
        browserAPI.notifications.create({
          type: 'basic',
          iconUrl: 'icon.png',
          title: '❌ Save Failed',
          message: error.message,
          priority: 2
        });
      });
  }
});

// Initialize on startup
browserAPI.runtime.onStartup.addListener(() => {
  captureCount = 0;
  updateBadge();
  resetBadgeDaily();
});

// ============================================================================
// SMART END-OF-CHAT ANALYSIS
// Analyzes conversation to extract valuable memories worth saving
// ============================================================================

/**
 * Analyze a conversation and extract valuable memories
 * Uses GPT to identify decisions, code, advice, todos, and facts
 */
async function analyzeConversationForMemories(data) {
  const { messages, platform, url } = data;
  const userId = await getUserId();

  if (!userId) {
    throw new Error('User not set up');
  }

  // Combine messages into conversation text
  const conversationText = messages.map((m) => `${m.role}: ${m.content}`).join('\n\n');

  // Call Haiven API to analyze conversation
  const response = await fetch(`${HAIVEN_API}/api/memory/analyze-conversation`, {
    method: 'POST',
    headers: await getAuthHeaders(),
    body: JSON.stringify({
      userId,
      conversation: conversationText,
      platform,
      url
    })
  });

  if (!response.ok) {
    const error = await response.json();
    throw new Error(error.message || 'Analysis failed');
  }

  const result = await response.json();

  // Check for duplicates for each suggested memory
  if (result.suggestions && result.suggestions.length > 0) {
    const suggestionsWithDupeCheck = await Promise.all(
      result.suggestions.map(async (suggestion) => {
        const dupeCheck = await checkForDuplicateMemory({
          content: suggestion.summary,
          userId
        });
        return {
          ...suggestion,
          isDuplicate: dupeCheck.isDuplicate,
          similarMemory: dupeCheck.similarMemory
        };
      })
    );
    result.suggestions = suggestionsWithDupeCheck;
  }

  return result;
}

/**
 * Check if a memory already exists (semantic similarity check)
 */
async function checkForDuplicateMemory(data) {
  const { content, userId: providedUserId } = data;
  const userId = providedUserId || (await getUserId());

  if (!userId) {
    return { isDuplicate: false };
  }

  try {
    const response = await fetch(`${HAIVEN_API}/api/memory/check-duplicate`, {
      method: 'POST',
      headers: await getAuthHeaders(),
      body: JSON.stringify({
        userId,
        content
      })
    });

    if (!response.ok) {
      return { isDuplicate: false };
    }

    return await response.json();
  } catch (error) {
    console.error('Duplicate check failed:', error);
    return { isDuplicate: false };
  }
}

/**
 * Save multiple memories at once
 * Includes deduplication to prevent saving identical content
 */
async function batchSaveMemories(data) {
  const { memories, platform, url } = data;
  const userId = await getUserId();

  if (!userId) {
    throw new Error('User not set up');
  }

  // DEDUPLICATION: Track what we've saved in this batch
  const savedContentHashes = new Set();
  const results = [];

  for (const memory of memories) {
    try {
      const content = memory.content || memory.summary;

      // Create simple hash of content to detect duplicates in this batch
      const contentHash = btoa(content.substring(0, 100)).substring(0, 20);

      if (savedContentHashes.has(contentHash)) {
        console.log('🧠 Haiven: Skipping duplicate in batch');
        results.push({ success: true, skipped: true, reason: 'duplicate_in_batch' });
        continue;
      }

      savedContentHashes.add(contentHash);

      const result = await captureMemory({
        content: content,
        source: platform || 'end-of-chat',
        url,
        tags: memory.tags || [memory.category],
        explicit: true,
        metadata: {
          category: memory.category,
          confidence: memory.confidence,
          why_save: memory.why_save
        }
      });
      results.push({ success: true, memory: result });
    } catch (error) {
      results.push({ success: false, error: error.message });
    }
  }

  // Update badge - only count actually saved (not skipped)
  const savedCount = results.filter((r) => r.success && !r.skipped).length;
  captureCount += savedCount;
  updateBadge();

  return { results, savedCount };
}
